import sys
import asyncio
import threading
import time
import os
import traceback
import gpiod
from enum import IntEnum
from .sensing_mgr.log import Log
from .sensing_mgr.io.led_device import LedDevice
from .cloud_agent.cloud_agent import CloudAgent

class ExitCode(IntEnum):
    SUCCESS = 0
    RESTART = 255 # gw_app will restart on exit() with this code

class GwApp:
    def __init__(self, loop, dev_cls, config_cls, config_path,
                 cloud_agent_file_path):

        model_config = config_cls()
        if not model_config.load(config_path):
            return

        conf = model_config.get_log_conf()

        conf = model_config.get_default_conf()
        cloud_config = conf.get(model_config.CLOUD_CONFIG)
        send_cloud = conf.get(model_config.SEND_CLOUD)
        valid_cache = conf.get(model_config.CACHE)

        model_device = dev_cls(model_config, cloud_config, send_cloud)
        device_id = os.environ.get('DEVINFO_SN')
        if send_cloud:
            cloud_agent = CloudAgent(loop, cloud_agent_file_path, device_id)
        else:
            cloud_agent = None

        if os.environ.get('SE050_NO_PM') == 'true':
            config = gpiod.line_request()
            config.request_type = gpiod.line_request.DIRECTION_OUTPUT
            gpiod_line = gpiod.chip(3).get_line(18)
            gpiod_line.request(config)
            gpiod_line.set_value(1 if send_cloud else 0)

        model_device.setup_reporters(model_config, cloud_agent, valid_cache,
                                     loop)
        self._device_id = device_id
        self._cloud_config = cloud_config
        self._send_cloud = send_cloud
        self._model_config = model_config
        self._model_device = model_device
        self._cloud_agent = cloud_agent

        self._loop = loop
        self._connection_task = None
        self._cloud_connected = False
        self._stop_reconnect = False

    def is_data_oneshot_stop(self):
        conf = self._model_config.get_default_conf()
        return conf.get(self._model_config.DATA_SEND_ONESHOT)

    def wait_container_stop_interval(self):
        conf = self._model_config.get_default_conf()
        return conf.get(self._model_config.WAIT_CONTAINER_STOP)

    async def oneshot(self, stop_interval):
        reporter_manager = self._model_device.reporter_manager()
        max_interval = reporter_manager.max_report_interval()
        send_margin = 5
        log = Log.instance()
        log.info("container exec data_oneshot_stop.")

        reporter_manager.set_oneshot()
        self._model_device.report_repository().set_oneshot()
        self.start()

        log.info(f"container waiting time {max_interval}[s].")
        await asyncio.sleep(max_interval + send_margin)

        start_time = time.time()
        log.info(f"container will stop after {stop_interval}[s].")
        await asyncio.sleep(stop_interval)

    async def send_deviceinformation(self):
        send_prop = {}
        device_data = {}
        device_data["DevInfo_SerialNumber"] = self._device_id
        # device_data["DevInfo_IMEI"] = os.environ.get('DEVINFO_IMEI')
        device_data["DevInfo_LAN_MAC_Addr"] = os.environ.get('DEVINFO_LAN_MAC')
        device_data["DevInfo_ABOS_Ver"] = os.environ.get('DEVINFO_ABOS_VER')
        device_data["DevInfo_Container_Ver"] = os.environ.get(
            'DEVINFO_CONTAINER_IMG_VER')
        send_prop["property"] = device_data
        return await self._cloud_agent.send(send_prop)

    async def reconnect(self):
        log = Log.instance()

        while not self._stop_reconnect:
            if not self._cloud_connected:
                if await self._cloud_agent.connect():
                    with LedDevice.get("app") as led:
                        led.on_or_off(True)
                    self._cloud_connected = True
                    self._cloud_agent.set_errcb(self.on_error)
            await asyncio.sleep(10)
        log.info("finished reconnect()")

    async def ready(self):
        log = Log.instance()

        if self._send_cloud:
            if await self._cloud_agent.connect():
                with LedDevice.get("app") as led:
                    led.on_or_off(True)
                self._cloud_connected = True
                self._cloud_agent.set_errcb(self.on_error)
            else:
                self._connection_task = self._loop.create_task(
                    self.reconnect())
            log.info("send device information to cloud.")
            if not await self.send_deviceinformation():
                with LedDevice.get("app") as led:
                    led.blink_fast()
        return

    def start(self):
        self._threads = []

        for loopable in self._model_device.loopables():
            thread = threading.Thread(target=loopable.start_loop)
            thread.start()
            self._threads.append(thread)
        return True

    def stop_threads(self):
        for loopable in self._model_device.loopables():
            loopable.request_stop()
        for thread in self._threads:
            thread.join()

    async def shutdown(self):
        log = Log.instance()
        self._stop_reconnect = True

        if self._connection_task:
            await self._connection_task
        if self._cloud_agent:
            await self._cloud_agent.disconnect()
        self.stop_threads()
        log.info(
            "all threads are quited, then shutdown the IoT client...")

    def on_error(self, msg):
        log = Log.instance()

        if msg == "ConnectionError":
            log.error("Connection error has occurred. Restarting gw_app....")
            sys.stdout.flush()
            sys.stderr.flush()
            os._exit(ExitCode.RESTART)

def stdin_listener():
    while True:
        selection = input("Press Q to quit\n")
        if (selection == 'Q' or selection == 'q'):
            print("Quitting...")
            break


async def mainloop(dev_cls, config_cls, config_path, cloud_agent_file_path):
    is_deamon = False
    args = sys.argv
    log = Log.instance()

    log.init_logging(config_path)

    try:
        with LedDevice.get("app") as led:
            led.blink_slow()

        if len(args) == 2:
            if args[1] == "-d":
                is_deamon = True

        loop = asyncio.get_event_loop()
        app = GwApp(
            loop, dev_cls, config_cls, config_path, cloud_agent_file_path)

        await app.ready()

        if app.is_data_oneshot_stop() is True:
            await app.oneshot(app.wait_container_stop_interval())
            await app.shutdown()
            return

        if not app.start():
            return

        if not is_deamon:
            user_finished = loop.run_in_executor(None, stdin_listener)
            await user_finished
            await app.shutdown()
        else:
            while True:
                await asyncio.sleep(10)
    except Exception as e:
        log.error("gw_app has error.")
        log.error(e.args)
        log.error(traceback.format_exc())
        log.preserved_error_log()
        return

#
# End of File
#
