# Copyright 2010-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
#
# This file is licensed under the Apache License, Version 2.0 (the "License").
# You may not use this file except in compliance with the License. A copy of the
# License is located at
#
# http://aws.amazon.com/apache2.0/
#
# This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS
# OF ANY KIND, either express or implied. See the License for the specific
# language governing permissions and limitations under the License.
#
# ABOUT THIS PYTHON SAMPLE: This sample is part of the AWS General Reference 
# Signing AWS API Requests top available at
# https://docs.aws.amazon.com/general/latest/gr/sigv4-signed-request-examples.html
#

import sys, os, base64, datetime, hashlib, hmac, json
import requests # pip install requests
import logging

class AWSRESTAPI:
    def __init__(self, access_key, secret_key, host, region, endpoint):
        self.access_key = access_key
        self.secret_key = secret_key
        self.host = host
        self.region = region
        self.endpoint = endpoint
        self._logging = logging.getLogger('cloud-agent')

    """
    def dataSet(self, method, service, host):
        self.method = method
        self.service = service
    """

    def request(self, method, service, body, uri, content_type=None, principal=None):
        amzdate, datestamp = self.__createDate()

        # *** Create Canonical Request
        # Step 1 is to define the verb (GET, POST, etc.)--already done.

        # Step 2: Create canonical URI--the part of the URI from domain to query
        # string (use '/' if no path)
        canonical_uri = ''

        # Step 3: Create the canonical query string. In this example (a GET request),
        # request parameters are in the query string. Query string values must
        # be URL-encoded (space=%20). The parameters must be sorted by name.
        # For this example, the query string is pre-formatted in the request_parameters variable.
        canonical_querystring = uri

        # Step 4: Create the canonical headers and signed headers. Header names
        # must be trimmed and lowercase, and sorted in code point order from
        # low to high. Note that there is a trailing \n.
        if content_type is None:
            canonical_headers = 'host:' + self.host + '\n' + 'x-amz-date:' + amzdate + '\n'
        else:
            canonical_headers = 'content-type:' + content_type + '\n'  + 'host:' + self.host + '\n' + 'x-amz-date:' + amzdate + '\n'

        # Step 5: Create the list of signed headers. This lists the headers
        # in the canonical_headers list, delimited with ";" and in alpha order.
        # Note: The request can include any headers; canonical_headers and
        # signed_headers lists those that you want to be included in the
        # hash of the request. "Host" and "x-amz-date" are always required.
        if content_type is None:
            signed_headers = 'host;x-amz-date'
        else:
            signed_headers = 'content-type;host;x-amz-date'

        # Step 6: Create payload hash (hash of the request body content). For GET
        # requests, the payload is an empty string ("").
        payload_hash = hashlib.sha256(body.encode('utf-8')).hexdigest()

        # Step 7: Combine elements to create canonical request
        canonical_request = method + '\n' + canonical_querystring + '\n\n' + canonical_headers + '\n' + signed_headers + '\n' + payload_hash

        # *** Create the String to Sign
        # Match the algorithm to the hashing algorithm you use, either SHA-1 or
        # SHA-256 (recommended)
        algorithm = 'AWS4-HMAC-SHA256'
        credential_scope = datestamp + '/' + self.region + '/' + service + '/' + 'aws4_request'
        string_to_sign = algorithm + '\n' +  amzdate + '\n' +  credential_scope + '\n' +  hashlib.sha256(canonical_request.encode('utf-8')).hexdigest()

        # *** Caluculate the signature
        # Create the signing key using the function defined above.
        signing_key = self.__getSignatureKey(self.secret_key, datestamp, self.region, service)

        # Sign the string_to_sign using the signing_key
        signature = hmac.new(signing_key, (string_to_sign).encode('utf-8'), hashlib.sha256).hexdigest()

        # Add signing information to the request
        # The signing information can be either in a query string value or in
        # a header named Authorization. This code shows how to use a header.
        # Create authorization header and add to request headers
        authorization_header = algorithm + ' ' + 'Credential=' + self.access_key + '/' + credential_scope + ', ' +  'SignedHeaders=' + signed_headers + ', ' + 'Signature=' + signature

        # The request can include any headers, but MUST include "host", "x-amz-date",
        # and (for this scenario) "Authorization". "host" and "x-amz-date" must
        # be included in the canonical_headers and signed_headers, as noted
        # earlier. Order here is not significant.
        # Python note: The 'host' header is added automatically by the Python 'requests' library.

        if content_type is None and principal is None:
            headers = {'x-amz-date':amzdate, 'Authorization':authorization_header}
        elif content_type is not None:
            headers = {'Content-Type':content_type, 'x-amz-date':amzdate, 'Authorization':authorization_header}
        elif principal is not None:
            headers = {'x-amzn-principal':principal, 'x-amz-date':amzdate, 'Authorization':authorization_header}

        # ************* SEND THE REQUEST *************
        request_url = self.endpoint + canonical_querystring

        self._logging.info('Request URL = %s', request_url)

        try:
            r = requests.request(method, request_url, headers=headers, data=body)

            self._logging.info('Response code: %d\n' % r.status_code)
            self._logging.info(r.text)
        except:
            raise Exception('request error')

        return r

    def __createDate(self):
        t = datetime.datetime.utcnow()
        amzdate = t.strftime('%Y%m%dT%H%M%SZ')
        datestamp = t.strftime('%Y%m%d') # Date w/o time, used in credential scope
        return amzdate, datestamp

    def __sign(self, key, msg):
        return hmac.new(key, msg.encode('utf-8'), hashlib.sha256).digest()

    def __getSignatureKey(self, key, dateStamp, regionName, serviceName):
        kDate = self.__sign(('AWS4' + key).encode('utf-8'), dateStamp)
        kRegion = self.__sign(kDate, regionName)
        kService = self.__sign(kRegion, serviceName)
        kSigning = self.__sign(kService, 'aws4_request')

        return kSigning

#
# End of File
#
