# Copyright 2010-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
#
# This file is licensed under the Apache License, Version 2.0 (the "License").
# You may not use this file except in compliance with the License. A copy of the
# License is located at
#
# http://aws.amazon.com/apache2.0/
#
# This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS
# OF ANY KIND, either express or implied. See the License for the specific
# language governing permissions and limitations under the License.
#
# ABOUT THIS PYTHON SAMPLE: This sample is part of the AWS General Reference 
# Signing AWS API Requests top available at
# https://docs.aws.amazon.com/general/latest/gr/sigv4-signed-request-examples.html
#

import sys, os, base64, datetime, hashlib, hmac, json
import requests # pip install requests
import logging
from .rest_api import AWSRESTAPI

class AwsDeviceProvisioning:
    def __init__(self, thing_name, config_data_cls):
    #def __init__(self, thing_name, config_data):
        self.thing_name = thing_name
        self.config_data = config_data_cls
        conf = self.config_data.get_prov_conf()

        self._logging = logging.getLogger('cloud-agent')

        if (not conf[self.config_data.ACCESS_KEY] or
            not conf[self.config_data.SECRET_KEY] or
            not conf[self.config_data.IOT_ACCOUNTID]):
            self.prov_exec = False
        else:
            self.prov_exec = True
            self.restapi = AWSRESTAPI(conf[self.config_data.ACCESS_KEY],
                                      conf[self.config_data.SECRET_KEY],
                                      conf[self.config_data.IOT_HOST],
                                      conf[self.config_data.IOT_REGION],
                                      conf[self.config_data.IOT_ENDPOINT])

    def clearCredential(self):
        clear_list = [self.config_data.ACCESS_KEY,
                      self.config_data.SECRET_KEY,
                      self.config_data.IOT_ACCOUNTID]
        self.config_data.clear_prov_conf(clear_list)

    def describeThing(self, thing_name):
        method = 'GET'
        service = 'execute-api'
        body = ''
        uri = '/things/' + thing_name
        return self.restapi.request(method, service, body, uri)

    def createThing(self, thing_name):
        method = 'POST'
        service = 'execute-api'
        body = ''
        uri = '/things/' + thing_name
        return self.restapi.request(method, service, body, uri)

    def createPolicy(self, policyname, policyfile):
        ret = None

        method = 'POST'
        service = 'execute-api'
        f = open(policyfile, 'r')
        data = f.readlines()

        conf = self.config_data.get_prov_conf()
        region = conf[self.config_data.IOT_REGION]
        account_id = conf[self.config_data.IOT_ACCOUNTID]

        if len(data) > 0:
            body = '{'
            body += '"policyDocument": "'
            body += "".join(data).replace("${AWS_REGION}", region).replace("${AWS_ACCOUNTID}", account_id).replace("\n", "").replace("\"", "\\\"").replace(" ", "")
            body += '"}'
            uri = '/policies/' + policyname
            ret = self.restapi.request(method, service, body, uri, content_type='application/json')

        return ret

    def registerCertificateWithoutCA(self, certfile):
        ret = None

        method = 'POST'
        service = 'execute-api'
        f = open(certfile, 'r')
        data = f.readlines()
        if len(data) > 0:
            body = '{'
            body += '"certificatePem": "'
            body += "".join(data).replace("\n", "\\n")
            body += '",'
            body += '"status": "ACTIVE"'
            body += '}'
            uri = '/certificate/register-no-ca'
            ret = self.restapi.request(method, service, body, uri, content_type='application/json')

        return ret

    def attachPolicy(self, policyname, certarn):
        method = 'PUT'
        service = 'execute-api'
        body = '{'
        body += '"target": "'
        body += certarn
        body += '"}'
        uri = '/target-policies/' + policyname
        return self.restapi.request(method, service, body, uri, content_type='application/json')

    def attachPrincipal(self, thingname, certarn):
        method = 'PUT'
        service = 'execute-api'
        body = ''
        uri = '/things/' + thingname + '/principals'
        principal = certarn
        return self.restapi.request(method, service, body, uri, principal=principal)

    def provisioning(self):

        try:
            if not self.prov_exec:
                self._logging.info("skip Device provisioning.")
                return True

            ret = self.describeThing(self.thing_name)
            if ret.status_code == 200:
                self._logging.info(" Device found.")
                return True

            ret = self.createThing(self.thing_name)
            if ret.status_code != 200 and ret.status_code != 409:
                return False

            conf = self.config_data.get_prov_conf()
            policy_name = self.thing_name + "_policy"
            ret = self.createPolicy(policy_name, conf[self.config_data.IOT_POLICY_FILE])
            if ret.status_code != 200 and ret.status_code != 409:
                return False

            ret = self.registerCertificateWithoutCA(conf[self.config_data.IOT_CERT_FILE])
            if ret.status_code == 200:
                param = json.loads(ret.text)
                certArn = param['certificateArn']

                self.attachPolicy(policy_name, certArn)
                self.attachPrincipal(self.thing_name, certArn)
            elif ret.status_code == 409:
                param = json.loads(ret.text)
                certArn = param['resourceArn']
                self.attachPrincipal(self.thing_name, certArn)
            else:
                return False
        except:
            self._logging.error(" Provisioning error.")
            return False

        return True

#
# End of File
#
