import configparser
import os
import logging

class AwsConfigConst:
    IOT_PORT_DEFAULT = 443
    IOT_CA_FILE_DEFAULT = "/cert/AmazonRootCA1.pem"
    IOT_CERT_FILE_DEFAULT = "/cert/device_cert.pem"
    IOT_PKCS11_PATH_DEFAULT = "/usr/lib/plug-and-trust/libsss_pkcs11.so"
    IOT_KEY_LABEL_DEFAULT = "sss:100100F0"
    IOT_POLICY_FILE_DEFAULT = "/config/aws_iot_policy.json"

class AwsConfig:
    CLOUD_NAME = "AWS"
    IOT_SHADOW_ENDPOINT = "AWS_IOT_SHADOW_ENDPOINT"
    IOT_PORT = "AWS_IOT_PORT"
    IOT_CA_FILE = "AWS_IOT_CA_FILE"
    IOT_CERT_FILE = "AWS_IOT_CERT_FILE"
    IOT_PKCS11_PATH = "AWS_IOT_PKCS11_PATH"
    IOT_KEY_LABEL = "AWS_IOT_KEY_LABEL"
    IOT_PIN = "AWS_IOT_PIN"
    ACCESS_KEY = "AWS_ACCESS_KEY"
    SECRET_KEY = "AWS_SECRET_KEY"
    IOT_HOST = "AWS_IOT_HOST"
    IOT_REGION = "AWS_IOT_REGION"
    IOT_ENDPOINT = "AWS_IOT_ENDPOINT"
    IOT_ACCOUNTID = "AWS_IOT_ACCOUNTID"
    IOT_POLICY_FILE = "AWS_IOT_POLICY_FILE"

    def __init__(self, config_path):
        self._configs = {}
        self._isConfigValid = True
        self._configpath = config_path
        self._logging = logging.getLogger('cloud-agent')

    def get_and_valid_data(self, config_data, dict_data, name, str=True, require=True, file=False):
        try:
            if str:
                get_data = config_data.get(name)
            else:
                get_data = config_data.getint(name)
        except Exception as e:
            self._logging.error("config %s get error.", name)
            self._logging.error("%s : %s", name, e.args)
            get_data = None

        if not get_data:
            if require:
                self._logging.error("%s is not found.", name)
                self._isConfigValid = False
            else:
                self._logging.warning("%s is not found.", name)
        else:
            dict_data[name] = get_data

        if file:
           if not os.path.isfile(dict_data[name]):
               self._logging.error("%s %s is not existed.", name, dict_data[name])
               self._isConfigValid = False

    def load(self, config_data):
        # Shadow
        shadow_conf = { AwsConfig.IOT_SHADOW_ENDPOINT : "",
                        AwsConfig.IOT_PORT : AwsConfigConst.IOT_PORT_DEFAULT,
                        AwsConfig.IOT_CA_FILE : AwsConfigConst.IOT_CA_FILE_DEFAULT,
                        AwsConfig.IOT_CERT_FILE : AwsConfigConst.IOT_CERT_FILE_DEFAULT,
                        AwsConfig.IOT_PKCS11_PATH : AwsConfigConst.IOT_PKCS11_PATH_DEFAULT,
                        AwsConfig.IOT_KEY_LABEL : AwsConfigConst.IOT_KEY_LABEL_DEFAULT,
                        AwsConfig.IOT_PIN : "" }

        self.get_and_valid_data(config_data, shadow_conf, AwsConfig.IOT_SHADOW_ENDPOINT)
        self.get_and_valid_data(config_data, shadow_conf, AwsConfig.IOT_PORT,
                                str=False, require=False)
        self.get_and_valid_data(config_data, shadow_conf, AwsConfig.IOT_CA_FILE,
                                require=False, file=True)
        self.get_and_valid_data(config_data, shadow_conf, AwsConfig.IOT_CERT_FILE,
                                require=False, file=True)
        self.get_and_valid_data(config_data, shadow_conf, AwsConfig.IOT_PKCS11_PATH,
                                require=False)
        self.get_and_valid_data(config_data, shadow_conf, AwsConfig.IOT_KEY_LABEL,
                                require=False)
        self.get_and_valid_data(config_data, shadow_conf, AwsConfig.IOT_PIN,
                                require=False)
        self._configs['shadow'] = shadow_conf

        # Prov
        prov_conf = { AwsConfig.ACCESS_KEY : "",
                      AwsConfig.SECRET_KEY : "",
                      AwsConfig.IOT_ACCOUNTID : "",
                      AwsConfig.IOT_HOST : "",
                      AwsConfig.IOT_REGION : "",
                      AwsConfig.IOT_ENDPOINT : "",
                      AwsConfig.IOT_POLICY_FILE : AwsConfigConst.IOT_POLICY_FILE_DEFAULT,
                      AwsConfig.IOT_CERT_FILE : AwsConfigConst.IOT_CERT_FILE_DEFAULT }

        self.get_and_valid_data(config_data, prov_conf, AwsConfig.ACCESS_KEY, require=False)
        self.get_and_valid_data(config_data, prov_conf, AwsConfig.SECRET_KEY, require=False)
        self.get_and_valid_data(config_data, prov_conf, AwsConfig.IOT_ACCOUNTID, require=False)

        if (prov_conf[AwsConfig.ACCESS_KEY] and
            prov_conf[AwsConfig.SECRET_KEY] and
            prov_conf[AwsConfig.IOT_ACCOUNTID]):
            prov_conf_error = True
        else:
            self._logging.info("Config parameter required for provisioning are not set.")
            prov_conf_error = False

        self.get_and_valid_data(config_data, prov_conf, AwsConfig.IOT_HOST, require=prov_conf_error)
        self.get_and_valid_data(config_data, prov_conf, AwsConfig.IOT_REGION, require=prov_conf_error)
        self.get_and_valid_data(config_data, prov_conf, AwsConfig.IOT_ENDPOINT, require=prov_conf_error)

        self.get_and_valid_data(config_data, prov_conf, AwsConfig.IOT_POLICY_FILE,
                                require=False, file=True)
        self.get_and_valid_data(config_data, prov_conf, AwsConfig.IOT_CERT_FILE,
                                require=False, file=True)
        self._configs['prov'] = prov_conf

    def clear_prov_conf(self, clear_list):
        config = configparser.ConfigParser()
        config.optionxform = str
        config.read(self._configpath)

        for item in clear_list:
            if self._configs['prov'][item]:
                config[AwsConfig.CLOUD_NAME][item] = ""

        with open(self._configpath, "w", encoding="utf-8") as configfile:
            config.write(configfile, True)

    def get_shadow_conf(self):
        return self._configs['shadow']

    def get_prov_conf(self):
        return self._configs['prov']

    def is_valid(self):
        return self._isConfigValid

#
# End of File
#
