import sys
import asyncio
import threading
import urllib.request
import logging

from .device_provisioning import AwsDeviceProvisioning
from .device_shadow import AwsDeviceShadow
from .config import AwsConfig
from ..connection_watchdog import ConnectionWatchdog

class AWSIoT:
    def __init__(self, loop, config_data, config_path, device_id):
        self._device_id = device_id

        self._config = AwsConfig(config_path)
        self._config.load(config_data)
        self._configpath = config_path

        self._device_shadow = AwsDeviceShadow(self._device_id, self._config)
        self._device_provisioning = AwsDeviceProvisioning(self._device_id, self._config)
        self._eventloop = loop
        self._sendDataList = {}

        self._logging = logging.getLogger('cloud-agent')
        self._watchdog = ConnectionWatchdog(self._eventloop, self._device_shadow.is_send_enable, 10)

    def setcb(self, cb_senddata, cb_recvdata):
        self._cbSendData = cb_senddata
        self._cbRecvData = cb_recvdata

    def set_errcb(self, cb_err):
        self._cbError = cb_err

    async def connect(self):
        self._logging.info("AWS connect start")

        if not self._config.is_valid():
            self._logging.error("Config error")
            return False

        if not self._device_provisioning.provisioning():
            self._logging.error("Device provisioning error")
            self._logging.error("AWS connect error")
            return False
        else:
            self._logging.info("Device provisioning success")
            self._device_provisioning.clearCredential()

        if not self._device_shadow.connect():
            self._logging.error("AWS connect error")
            return False

        self._device_shadow.create_subscribe_update_shadow_accepted_callback(self.on_update_shadow_accepted)
        self._device_shadow.create_subscribe_shadow_delta_updated_callback(self.on_shadow_delta_updated)
        self._device_shadow.create_subscribe_get_shadow_accepted_callback(self.on_get_shadow_accepted)
        self._device_shadow.publish_get_shadow()

        # logging example
        self._logging.info("AWS connect complete")
        self._watchdog.start()

        return True

    async def send(self, message):

        send_data = {}
        if 'data' in message:
            data = message['data']
            send_data.update(data)
        if 'property' in message:
            property = message['property']
            send_data.update(property)

        ret = self._device_shadow.update_publish_shadow(send_data)
        if not ret:
            list_msg = []
            for data in self._sendDataList.values():
                list_msg.append(data)
            self._sendDataList.clear()
            list_msg.append(message)
            self._cbSendData(False, list_msg)
        else:
            self._logging.info("Publish client token: %s", ret)
            self._sendDataList[ret] = message
        return True

    def on_update_shadow_accepted(self, response):
        try:
            # check that this is a response to a request from this session
            try:
                #self._logging.info("Finished updating desired shadow value to '{}'.".format(response.state.desired))
                self._logging.info("Finished updating reported shadow value to '{}'.".format(response.state.reported))
                self._logging.info("Accepted client token: %s", response.client_token)
                try:
                    del self._sendDataList[response.client_token]
                except KeyError:
                    self._logging.info("Client token not in list : %s", response.client_token)
            except:
                exit("Updated shadow is missing the target property.")
        except Exception as e:
            exit(e)

    def on_get_shadow_accepted(self, response):
        try:
            # check that this is a response to a request from this session
            try:
                self._logging.info("Finished get shadow value to '{}'.".format(response.state.desired))
                ret = self._cbRecvData(response.state.desired)
                if ret == True and response.state.desired is not None:
                    self._device_shadow.update_publish_shadow(response.state.desired)
            except:
                exit("Get shadow is missing the target property.")
        except Exception as e:
            exit(e)

    def on_shadow_delta_updated(self, delta):
        self._logging.info("Delta reported".format(delta.state))
        ret = self._cbRecvData(delta.state)
        if ret == True:
            self._device_shadow.update_publish_shadow(delta.state)

    async def disconnect(self):
        self._logging.info("all threads are quited, then shutdown the IoT client...")
        self._device_shadow.disconnect()
        self._watchdog.stop()

#
# End of File
#
