import asyncio
from .modules.lib.report_repository import ReportRepository
from .cache import Cache
from .log import Log


class CloudReportRepository(ReportRepository):

    def __init__(self, report_queue, alarm_queue, cloud_agent, valid_cache,
                 send_cloud, loop):
        super().__init__(report_queue, alarm_queue)
        self._loop = loop
        self._cloud_agent = cloud_agent
        self._oneshot = False
        self._valid_cache = valid_cache
        self._cache = Cache()
        self._send_cloud = send_cloud

    def finish(self):
        if self._valid_cache is True:
            self._cache.flush()
        else:
            self._cache.clear()
            self._cache.remove()

    def process_report(self, report):
        reported_data = {}
        reported_data["data"] = report.reported_data
        reported_data["data"]["timestamp"] = report.reported_at
        self._send_cloud_agent(reported_data)
        return True

    def process_reports(self, reports):
        if len(reports) == 1:
            return self.process_report(reports[0])
        else:
            measurements = []
            for report in reports:
                if report:
                    reported_data = {}
                    reported_data["data"] = report.reported_data
                    reported_data["data"]["timestamp"] = report.reported_at
                    measurements.append(reported_data)
            if measurements:
                self._send_cloud_agent(measurements)
                return True
            else:
                return False

    def process_alarm(self, alarm):
        return True

    def dispatch_operation(self):
        num = 1
        if self._cache.empty() is True:
            return False
        result = self._cache.empty()

        while result is False and num > 0:
            dict = self._cache.pop()
            if dict is not None:
                self._send_cloud_agent(dict)
            num -= 1
        return False

    def _send_cloud_agent(self, send_dict):
        log = Log.instance()
        log.info(send_dict)

        if self._send_cloud:
            future = asyncio.run_coroutine_threadsafe(
                self._cloud_agent.send(send_dict), self._loop
            )
            try:
                future.result(10)
            except asyncio.TimeoutError:
                log.error('The coroutine took too long, cancelling the task...')
                future.cancel()
            except Exception as exc:
                log.error(f'The coroutine rased an exception: {exc!r}.: {str(type(exc))}')
                future.cancel()

    def set_oneshot(self):
        self.set_interval(1)

    def on_send_cloud_agent(self, result, data):
        if data is None:
            return

        for dict in data:
            self._cache.put(dict)

#
# EOF
#
