import time
from pymodbus.client import ModbusSerialClient as ModbusClient
from pymodbus.exceptions import ModbusIOException
from ..log import Log


class RS485_Mgr:
    DATA1 = 1
    DATA2 = 2
    DATA3 = 3
    DATA4 = 4

    def __init__(self, port):
        self._port = port
        self._connection = {}

    def create_new_connection(
            self, method, baudrate, device_id, byte_size, stop_bits, parity):
        client = None

        if parity == 'none':
            parity = 'N'
        elif parity == 'odd':
            parity = 'O'
        elif parity == 'even':
            parity = 'E'

        client = ModbusClient(
            method=method,
            port=self._port,
            stopbits=stop_bits,
            bytesize=byte_size,
            parity=parity,
            baudrate=baudrate,
            timeout=1)

        self._connection[device_id] = client

    @staticmethod
    def get_bits_value(result, count, endian_isBig):
        return result.bits[0]

    @staticmethod
    def get_register_value(result, count, endian_isBig):
        if count == 1:
            return result.registers[0]
        elif count == 2:
            if endian_isBig is True:
                return (result.registers[0] << 16) + result.registers[1]
            else:
                return (result.registers[1] << 16) + result.registers[0]

    def read_func(
            self, client, device_id, reg_addr, func_code, count, endian_isBig):
        io_func = None
        val_func = None

        if func_code == 0x01:
            io_func = client.read_coils
            val_func = RS485_Mgr.get_bits_value
        elif func_code == 0x02:
            io_func = client.read_discrete_inputs
            val_func = RS485_Mgr.get_bits_value
        elif func_code == 0x03:
            io_func = client.read_holding_registers
            val_func = RS485_Mgr.get_register_value
        elif func_code == 0x04:
            io_func = client.read_input_registers
            val_func = RS485_Mgr.get_register_value
        else:
            return None

        result = io_func(address=reg_addr, count=count, slave=device_id)
        if type(result) == ModbusIOException:
            raise result
        return val_func(result, count, endian_isBig)

    def write_func(
            self,
            client, device_id, reg_addr, func_code, count, write_data):
        io_func = None
        if func_code == 0x05:
            io_func = client.write_coil
        elif func_code == 0x06:
            io_func = client.write_register
        elif func_code == 0x0F:
            io_func = client.write_coils
        elif func_code == 0x10:
            io_func = client.write_registers
        else:
            return None

        result = io_func(address=reg_addr, value=write_data, slave=device_id)
        if type(result) == ModbusIOException:
            raise result
        return result

    def write_command(
            self, method, baudrate, byte_size, stop_bits, parity,
            device_id, reg_addr, func_code, write_byte):
        log = Log.instance()

        if parity == 'none':
            parity = 'N'
        elif parity == 'odd':
            parity = 'O'
        elif parity == 'even':
            parity = 'E'

        client = ModbusClient(
            method=method,
            port=self._port,
            stopbits=stop_bits,
            bytesize=byte_size,
            parity=parity,
            baudrate=baudrate,
            timeout=1)

        try:
            if client.connect():
                if func_code > 0x04:
                    try:
                        self.write_func(
                            client, device_id, reg_addr,
                            func_code, 1, write_byte)
                    except ModbusIOException as e:
                        log.error("I/O error at RS485_Mgr.write_func(), do retry.")
                        self.write_func(
                            client, device_id, reg_addr,
                            func_code, 1, write_byte)
                        log.info("the retry succeed")
                client.close()
            else:
                log.error("RS485 client.connect() is error.")
            client.close()
        except Exception as e:
            log.error("RS485 client.connect() is error.")
            log.error(e.args)
            client.close()

    def read(self,
             device_id, reg_addr, func_code, count, endian_isBig):
        log = Log.instance()

        if device_id not in self._connection:
            return
        client = self._connection[device_id]

        try:
            if client.connect():
                result = None
                if func_code <= 0x04:
                    try:
                        result = self.read_func(
                            client, device_id, reg_addr, func_code,
                            count, endian_isBig)
                    except ModbusIOException:
                        log.error("I/O error at RS485_Mgr.read_func(), do retry...")
                        result = self.read_func(
                            client, device_id, reg_addr, func_code,
                            count, endian_isBig)
                        log.info("the retry succeed")
                client.close()
                return result
            else:
                log.error("RS485 client.connect() is error.")
            client.close()
        except Exception as e:
            log.error("RS485 client.connect() is error.")
            log.error(e.args)
            client.close()


#
# EOF
#
