import asyncio
import certifi
import json
import os
import platform
import subprocess
import configparser
import time,datetime
import logging
from enum import IntEnum

from azure.iot.device import X509
from azure.iot.device import Message, MethodResponse
from azure.iot.device.aio import IoTHubDeviceClient
from azure.iot.device.aio import ProvisioningDeviceClient
from azure.iot.device.exceptions import CredentialError

from .config import AzureConfig

class ErrorCode(IntEnum):
    Success = 1,
    AuthenticationFailed = 2,
    ConnectionFailed = 3,
    DisconnectFailed = 4

class ClientStatus(IntEnum):
    Idle = 1,
    TryingToConnect = 2,
    Connected = 3,
    TryingToDisconnect = 4,
    Error = 5

class AzureClient:
    def __init__(self, config_data, cb_property, cb_command):
        self._clientHandle = None
        self._clientStatus = ClientStatus.Idle

        self._cbProperty = cb_property
        self._cbCommand = cb_command

        self._config = AzureConfig()
        self._config.load(config_data)
        self._logging = logging.getLogger('cloud-agent')

        if platform.system() == "Linux":
            os.environ["SSL_CERT_FILE"] = certifi.where()

    def is_connected(self):
        return self._isConnected

    def judge_reconnect(self):
        if ((self._clientStatus == ClientStatus.Error) or
            (self._clientStatus != ClientStatus.TryingToConnect and
             not self._clientHandle.connected)):
            return True
        else:
            return False

    def process_alarm(self, alarm):
        return self._modelDev.process_alarm(alarm)

    async def _provision_device(self, provisioning_host, id_scope, registration_id, x509, model_id):
        provisioning_device_client = ProvisioningDeviceClient.create_from_x509_certificate(
            provisioning_host=provisioning_host,
            registration_id=registration_id,
            id_scope=id_scope,
            x509=x509,
        )

        try:
            ret = await provisioning_device_client.register()
        except:
            raise Exception('provisioing error')
        return ret

    async def auth_and_connect(self):
        if not self._config.is_valid():
            return ErrorCode.AuthenticationFailed

        if (self._clientStatus != ClientStatus.Idle and
            self._clientStatus != ClientStatus.Error):
            return ErrorCode.Success

        try:
            if (self._clientStatus == ClientStatus.Error and
                self._clientHandle != None and
                self._clientHandle.connected):
                await asyncio.wait_for(self._clientHandle.disconnect(), timeout=10)
        except:
            self._logging.info("caught an exception from disconnect().")
            self._clientStatus = ClientStatus.Error
            return ErrorCode.DisconnectFailed

        self._clientHandle = None
        self._clientStatus = ClientStatus.TryingToConnect

        config_data = self._config.get_conf()

        x509 = X509(
            cert_file = config_data[AzureConfig.IOT_CERT_FILE],
            key_file = config_data[AzureConfig.IOT_KEY_FILE],
        )

        try:
            registration_result = await self._provision_device(
                config_data[AzureConfig.IOT_DEVICE_DPS_ENDPOINT],
                config_data[AzureConfig.IOT_DEVICE_DPS_ID_SCOPE],
                config_data[AzureConfig.IOT_DEVICE_ID],
                x509,
                config_data[AzureConfig.IOT_MODEL_ID]
            )

            if registration_result.status == "assigned":
                self._logging.info("Device was assigned")
                self._logging.info("hub: %s", registration_result.registration_state.assigned_hub)
                self._logging.info("deviceid: %s", registration_result.registration_state.device_id)

                device_client = IoTHubDeviceClient.create_from_x509_certificate(
                    x509=x509,
                    hostname=registration_result.registration_state.assigned_hub,
                    device_id=registration_result.registration_state.device_id,
                    product_info=config_data[AzureConfig.IOT_MODEL_ID],
                    connection_retry=False
                )
            else:
                self._logging.error("Could not provision device. Aborting device connection.")
                self._clientStatus = ClientStatus.Error
                return ErrorCode.ConnectionFailed
        except:
            self._logging.error("Could not provision device. Aborting device connection.")
            self._clientStatus = ClientStatus.Error
            return ErrorCode.ConnectionFailed

        try:
            await device_client.connect()

            device_client.on_background_exception = self._background_exception
            device_client.on_connection_state_change = self._connection_state_change
            device_client.on_method_request_received = self._cbCommand
            device_client.on_twin_desired_properties_patch_received = self._cbProperty
            device_client.on__message_received = self._message_received_handler
            self._clientHandle = device_client
            self._clientStatus = ClientStatus.Connected

            twin = await device_client.get_twin()
            await self._cbProperty(twin)

            return ErrorCode.Success
        except:
            self._clientStatus = ClientStatus.Error
            return ErrorCode.ConnectionFailed

    async def disconnect(self):
        if self._clientStatus == ClientStatus.Connected:
            self._clientStatus = ClientStatus.TryingToDisconnect
            await self._clientHandle.disconnect()
            self._clientStatus = ClientStatus.Idle

    async def shutdown(self):
        await self.disconnect()
        if self._clientHandle is not None:
            await self._clientHandle.shutdown()

    async def send_telemetry(self, telemetry_data):
        if self._clientStatus != ClientStatus.Connected:
            return False

        if "$.sub" in telemetry_data:
            component = telemetry_data["$.sub"]
            del telemetry_data["$.sub"]
        else:
            component = None
        if "timestamp" in telemetry_data:
            epoch = telemetry_data["timestamp"]
            del telemetry_data["timestamp"]
            telemetry_data["epoch"] = epoch
        else:
            epoch = None
        msg = Message(json.dumps(telemetry_data))
        msg.content_enconding = "utf-8"
        msg.content_type      = "application/json"
        if component is not None:
            msg.custom_properties["$.sub"] = component
        if epoch is not None:
            msg.custom_properties["iothub-creation-time-utc"] = datetime.datetime.fromtimestamp(epoch).astimezone().isoformat()
        self._logging.info("Send data: {}".format(telemetry_data))
        try:
             await asyncio.wait_for(self._clientHandle.send_message(msg), timeout=10)
        except:
            self._logging.info("caught an exception from send_message().")
            self._clientStatus = ClientStatus.Error
            return False

        return True

    async def send_updated_prop(self, prop_data):
        if self._clientStatus != ClientStatus.Connected:
            return False
        try:
            await asyncio.wait_for(self._clientHandle.patch_twin_reported_properties(prop_data), timeout=10)
        except:
            self._logging.info("connection has broken.")
            self._clientStatus = ClientStatus.Error
            return False

        return True

    async def _do_connect(self):
        await self.auth_and_connect()

    async def send_command_response(self, result, method_request):
        status = 400
        if not result:
            self._logging.info("Could not execute the direct method: %s", method_request.name)
            payload = {"result": False, "data": "unknown method"}
        else:
            payload = {
                "result": True,
                "data": (method_request.name + " is succeeded" if isinstance(result, bool) else result)
            }
            status = 200
        method_response = MethodResponse.create_from_method_request(
            method_request, status, payload
        )

        await asyncio.wait_for(self._clientHandle.send_method_response(method_response), timeout=10)

    async def send_property_response(self, patch):
        ignore_keys = ["__t", "$version"]
        version = patch["$version"]
        props   = {}
        for name, value in patch.items():
            if not name in ignore_keys:
                props[name] = {
                    "ac": 200,
                    "ad": "Successfully executed patch",
                    "av": version,
                    "value": value
                }
        await asyncio.wait_for(self._clientHandle.patch_twin_reported_properties(props), timeout=10)

    async def _message_received_handler(self, msg):
        self._logging.info("got message from the cloud, msg= %s", msg)

    async def _background_exception(self, msg):
        self._logging.info("got exception, msg= %s", msg)

    async def _connection_state_change(self):
        self._logging.info("connection state change, flag = %d", self._clientHandle.connected)

#
# End of File
#

