import sys
import asyncio
import threading
import urllib.request
import logging

from .client import AzureClient
from .client import ErrorCode
from ..wdt import WatchdogTimer

class AzureIoT:
    def __init__(self, loop, config_data):
        azure_client = AzureClient(config_data, self._recv_property, self._recv_command)
        self._azure_client = azure_client
        self._doCheckConnection  = True
        self._eventloop = loop
        self._logging = logging.getLogger('cloud-agent')
        self._wdt = WatchdogTimer()

    def setcb(self, cb_senddata, cb_recvdata):
        self._cbProperty = cb_recvdata
        self._cbCommand = cb_recvdata
        self._cbSend = cb_senddata

    def set_errcb(self, cb_err):
        self._cbError = cb_err

    async def connect(self):
        self._logging.info("Azure IoT connect start")
        self._wdt.stop()
        result = await self._azure_client.auth_and_connect()

        if result == ErrorCode.AuthenticationFailed:
            self._logging.error("Could not connect to the Azure IoT.")
            return False
        elif result == ErrorCode.ConnectionFailed:
            self._logging.error("Could not connect to the Azure IoT.")
            return False

        if self._doCheckConnection:
            connection_task = self._eventloop.create_task(
                self.check_connection())
            self._wdt.start()

        self._logging.info("Azure IoT connect end")

        return True

    async def send(self, message):

        result = False
        if 'data' in message:
            result = await self._azure_client.send_telemetry(message['data'])
        if 'property' in message:
            result = await self._azure_client.send_updated_prop(message['property'])

        list_msg = []
        list_msg.append(message)

        if not result:
            self._cbSend(result, list_msg)

        return result

    async def _recv_command(self, method_request):
        self._logging.info("Recv command name: %s payload: %s", method_request.name, method_request.payload)
        data = { method_request.name : method_request.payload }
        result = self._cbCommand(data)
        await self._azure_client.send_command_response(result, method_request)
        return True

    async def _recv_property(self, patch):

        self._logging.info("Recv data: {}".format(patch))

        if 'desired' in patch:
            recv_data = patch['desired']
        else:
            recv_data = patch

        version = recv_data['$version']
        del (recv_data)['$version']
        result = self._cbProperty(recv_data)
        recv_data['$version'] = version
        await self._azure_client.send_property_response(recv_data)

        return True

    def stop_threads(self):
        for loopable in self._model_device.loopables():
            loopable.request_stop()
        for thread in self._threads:
            thread.join()
        self._wdt.stop()

    async def disconnect(self):
        self._wdt.stop()
        self._doCheckConnection = False
        self._logging.info("all threads are quited, then shutdown the IoT client...")
        await self._azure_client.shutdown()

    async def check_connection(self):
        while self._doCheckConnection:
            if self._azure_client.judge_reconnect():
                result = await self._azure_client.auth_and_connect()
                if result == ErrorCode.DisconnectFailed:
                    self._cbError("ConnectionError")
            self._wdt.reset()
            await asyncio.sleep(10)

#
# End of File
#
