/**
 * Copyright (C) 2023-2024 Atmark Techno, Inc. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 *
 * Note: constants come from mmc-utils, which is GPL-2.0, but
 * constants are not licensable.
 */

#include <errno.h>
#include <fcntl.h>
#include <linux/mmc/ioctl.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>
#include <unistd.h>

#include "mmc.h"
#include "agent_log.h"

// defines required for extcsd ioctl
#define MMC_SEND_EXT_CSD        8       /* adtc                         R1  */

#define MMC_RSP_SPI_S1  (1 << 7)                /* one status byte */
#define MMC_RSP_SPI_R1  (MMC_RSP_SPI_S1)
#define MMC_RSP_PRESENT (1 << 0)
#define MMC_RSP_CRC     (1 << 2)                /* expect valid crc */
#define MMC_RSP_OPCODE  (1 << 4)                /* response contains opcode */
#define MMC_RSP_R1      (MMC_RSP_PRESENT|MMC_RSP_CRC|MMC_RSP_OPCODE)
#define MMC_CMD_ADTC    (1 << 5)

#define EXT_CSD_REV                     192
#define EXT_CSD_PRE_EOL_INFO            267     /* RO */

#define EXT_CSD_SIZE 512

static int read_extcsd(const char *device, uint8_t *ext_csd)
{
    int fd = open(device, O_RDONLY);
    if (fd < 0)
    {
        AGENT_LOG_WARN("Could not open %s to get extcsd: %d", device, errno);
        return -1;
    }

    int ret = 0;
    struct mmc_ioc_cmd idata = { };
    // clear rev to make sure kernel wrote it
    ext_csd[EXT_CSD_REV] = 0;

    idata.opcode = MMC_SEND_EXT_CSD;
    idata.flags = MMC_RSP_SPI_R1 | MMC_RSP_R1 | MMC_CMD_ADTC;
    idata.blksz = EXT_CSD_SIZE;
    idata.blocks = 1;
    mmc_ioc_cmd_set_data(idata, ext_csd);

    ret = ioctl(fd, (int)MMC_IOC_CMD, &idata);
    if (ret)
        AGENT_LOG_WARN("Could not query %s for extcsd: %d", device, errno);

    close(fd);

    return ret;
}

int mmcGetPreEolInfo(const char *device)
{
    uint8_t ext_csd[EXT_CSD_SIZE];

    if (read_extcsd(device, ext_csd))
        return -1;

    // supported from MMC 5.0 onwards
    if (ext_csd[EXT_CSD_REV] < 7)
        return -1;

    return ext_csd[EXT_CSD_PRE_EOL_INFO];
}
