/**
 * Copyright (C) 2023-2024 Atmark Techno, Inc. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */

#include "job_handler.h"
#include "agent_log.h"
#include "power_notification_handler.h"

// constructor
JobHandler::JobHandler(
    String thingName,
    bool isCommandExecAllowed,
    std::shared_ptr<Mqtt::MqttConnection> connection)
    : thingName_(thingName)
    , isCommandExecAllowed_(isCommandExecAllowed)
    , connection_(connection)
{
    jobClient_ = new IotJobsClient(connection_);
}

JobHandler::~JobHandler(void)
{
    delete jobClient_;
}

void JobHandler::MainLoop(void)
{
    AGENT_LOG_DEBUG("start job_handler main loop");
    jobExecutionsChangedPromise_ = std::promise<void>();

    if (!initialQueuedJobs_.empty()) {
        RunJobsFromSummary(initialQueuedJobs_);
        initialQueuedJobs_.clear();
    }

    while (! isQuitRequested_)
    {
        Vector<JobExecutionSummary> inProgressJobs;
        Vector<JobExecutionSummary> queuedJobs;

        jobExecutionsChangedPromise_.get_future().wait();
        if (isQuitRequested_) {
            return;
        }
        jobExecutionsChangedPromise_ = std::promise<void>();

        if (!GetPendingJobExecutions(queuedJobs, inProgressJobs))
        {
            AGENT_LOG_WARN("failed to get Job executions");
        }

        // ignore IN_PROGRESS jobs
        if (!queuedJobs.empty())
        {
            RunJobsFromSummary(queuedJobs);
        }
    }
}

bool JobHandler::StartMainLoop(void)
{
    if (mainLoopThread_.get_id() != std::thread::id()) {
        AGENT_LOG_ERROR("the thread is already started.");
        return false;
    }
    mainLoopThread_ = std::thread(&JobHandler::MainLoop, this);
    return true;
}

void JobHandler::StopMainLoop(void)
{
    if (mainLoopThread_.get_id() != std::thread::id()) {
        isQuitRequested_ = true;
        jobExecutionsChangedPromise_.set_value();
        mainLoopThread_.join();
    }
}

bool JobHandler::Subscribe(void)
{
    bool ret = true;

    ret &= SubscribeJobExecutionsChanged();
    ret &= SubscribeUpdateJobExecution();
    ret &= SubscribeGetPendingJobExecutions();
    ret &= SubscribeDescribeJobExecution();

    return ret;
}

void JobHandler::SubscribeCanceledJob(String jobId)
{
    std::promise<void> subscribeJobCancelPromise;

    auto onCancelMesseageReceived = [&](
        Mqtt::MqttConnection &,
        const String &,
        const ByteBuf &payload,
        bool,
        Mqtt::QOS,
        bool)
    {
        String payloadStr(reinterpret_cast<char *>(payload.buffer), payload.len);
        JsonObject payloadJson(payloadStr);
        String cancelJobId = payloadJson.View().GetString("jobId");
        String pidStr = readFileContents(PID_FILE_PATH(cancelJobId));

        if (pidStr != "")
        {
            int pid = stoi_safe(pidStr);
            kill(pid, SIGKILL);
        }
        else
        {
            AGENT_LOG_WARN("Failed to kill canceled job process");
        }
        SetJobIdExecutingCommand("");
    };

    auto onCancelSubAck = [&](
        Mqtt::MqttConnection &,
        unsigned short int,
        const String &,
        Mqtt::QOS,
        int errCode)
    {
        AGENT_LOG_DEBUG("cancel sub ack! errCode: %d", errCode);
        subscribeJobCancelPromise.set_value();
    };

    String topic = "$aws/events/jobExecution/" + jobId + "/canceled";
    connection_->Subscribe(
        topic.c_str(),
        AWS_MQTT_QOS_AT_LEAST_ONCE,
        std::move(onCancelMesseageReceived),
        onCancelSubAck
    );

    subscribeJobCancelPromise.get_future().wait();
}

void JobHandler::UnsubscribeCanceledJob(String jobId)
{
    std::promise<void> unsubscribeJobCancelPromise;
    auto onOperationCompleteHandler = [&](
        Mqtt::MqttConnection &,
        unsigned short int,
        int errCode)
    {
        AGENT_LOG_DEBUG("cancel unsub completed! errCode: %d", errCode);
        unsubscribeJobCancelPromise.set_value();
    };

    String topic = "$aws/events/jobExecution/" + jobId + "/canceled";
    connection_->Unsubscribe(
        topic.c_str(),
        onOperationCompleteHandler
    );

    unsubscribeJobCancelPromise.get_future().wait();
}

void JobHandler::SetJobIdExecutingCommand(String jobId)
{
    std::lock_guard<std::mutex> lock(mtx_);
    jobIdExecutingCommand_ = jobId;
}

String JobHandler::GetJobIdExecutingCommand(void)
{
    String ret = "";
    std::lock_guard<std::mutex> lock(mtx_);
    ret = jobIdExecutingCommand_;
    return ret;
}

void JobHandler::SetJobIdExecutingSwupdate(String jobId)
{
    std::lock_guard<std::mutex> lock(mtx_);
    jobIdExecutingSwupdate_ = jobId;
}

String JobHandler::GetJobIdExecutingSwupdate(void)
{
    String ret = "";
    std::lock_guard<std::mutex> lock(mtx_);
    ret = jobIdExecutingSwupdate_;
    return ret;
}

bool JobHandler::SubscribeJobExecutionsChanged(void)
{
    bool ret = true;
    std::promise<void> subscribedPromise;

    JobExecutionsChangedSubscriptionRequest jobExecutionsChangedSubscriptionRequest;
    jobExecutionsChangedSubscriptionRequest.ThingName = thingName_;

    auto subscribedPromiseHandler = [&](int)
    {
        subscribedPromise.set_value();
    };

    auto onJobExecutionsChanged = [&](JobExecutionsChangedEvent *event, int ioErr)
    {
        if (ioErr)
        {
            AGENT_LOG_WARN("onJobExecutionsChanged: Error %d occurred", ioErr);
            return;
        }
        jobExecutionsChangedPromise_.set_value();
    };

    ret &= jobClient_->SubscribeToJobExecutionsChangedEvents(
        jobExecutionsChangedSubscriptionRequest,
        AWS_MQTT_QOS_AT_LEAST_ONCE,
        std::move(onJobExecutionsChanged),
        subscribedPromiseHandler);

    subscribedPromise.get_future().wait();

    return ret;
}

bool JobHandler::SubscribeUpdateJobExecution(void)
{
    bool ret = true;
    std::promise<void> subscribedAcceptedFuture;
    std::promise<void> subscribedRejectedFuture;

    UpdateJobExecutionSubscriptionRequest updateSubscriptionRequest;
    updateSubscriptionRequest.ThingName = thingName_;
    updateSubscriptionRequest.JobId = "+";

    auto subscribedAcceptedFutureHandler = [&](int)
    {
        subscribedAcceptedFuture.set_value();
    };

    auto onUpdateJobExecutionAccepted = [&](UpdateJobExecutionResponse *response, int ioErr)
    {
        finishUpdateJobStatePromise_.set_value(std::nullopt);
        if (ioErr)
        {
            AGENT_LOG_WARN("onUpdateJobExecutionAccepted: Error %d occurred", ioErr);
            return;
        }
        AGENT_LOG_DEBUG("Request to update job was accepted.");
    };

    ret &= jobClient_->SubscribeToUpdateJobExecutionAccepted(
        updateSubscriptionRequest,
        AWS_MQTT_QOS_AT_LEAST_ONCE,
        onUpdateJobExecutionAccepted,
        subscribedAcceptedFutureHandler
    );

    auto subscribedRejectedFutureHandler = [&](int)
    {
        subscribedRejectedFuture.set_value();
    };

    auto onUpdateJobExecutionRejected = [&](RejectedError *rejectedError, int ioErr)
    {
        finishUpdateJobStatePromise_.set_value(*rejectedError);
        if (ioErr)
        {
            AGENT_LOG_WARN("onUpdateJobExecutionRejected: Error %d occurred", ioErr);
            return;
        }

        if (rejectedError)
        {
            AGENT_LOG_WARN("Request to update job status was rejected. code:%d message:'%s'.",
                           (int)rejectedError->Code.value(),
                           rejectedError->Message->c_str()
            );
            return;
        }
    };

    ret &= jobClient_->SubscribeToUpdateJobExecutionRejected(
        updateSubscriptionRequest,
        AWS_MQTT_QOS_AT_LEAST_ONCE,
        onUpdateJobExecutionRejected,
        subscribedRejectedFutureHandler
    );

    subscribedAcceptedFuture.get_future().wait();
    subscribedRejectedFuture.get_future().wait();

    return ret;
}

bool JobHandler::SubscribeGetPendingJobExecutions(void)
{
    bool ret = true;
    std::promise<void> subscribedAcceptedPromise;
    std::promise<void> subscribedRejectedPromise;

    GetPendingJobExecutionsSubscriptionRequest getPendingJobExecutionsSubscriptionRequest;
    getPendingJobExecutionsSubscriptionRequest.ThingName = thingName_;

    auto onGetPendingJobExecutionsAccepted = [&](GetPendingJobExecutionsResponse *response, int ioErr)
    {
        Vector<JobExecutionSummary> queuedJobs;
        Vector<JobExecutionSummary> inProgressJobs;
        if (ioErr != AWS_OP_SUCCESS)
        {
            AGENT_LOG_WARN("Error getting job: %s.", ErrorDebugString(ioErr));
        }

        if (response->QueuedJobs.value().size() > 0)
        {
            queuedJobs = response->QueuedJobs.value();
        }

        if (response->InProgressJobs.value().size() > 0)
        {
            inProgressJobs = response->InProgressJobs.value();
        }
        getQueuedJobsPromise_.set_value(queuedJobs);
        getInProgressJobsPromise_.set_value(inProgressJobs);
    };

    auto onGetPendingJobExecutionsAcceptedSubAck = [&](int ioErr)
    {
        if (ioErr != AWS_OP_SUCCESS)
        {
            AGENT_LOG_WARN("Error subscribing to get Job accepted: %s", ErrorDebugString(ioErr));
            return;
        }
        subscribedAcceptedPromise.set_value();
    };

    ret &= jobClient_->SubscribeToGetPendingJobExecutionsAccepted(
        getPendingJobExecutionsSubscriptionRequest,
        AWS_MQTT_QOS_AT_LEAST_ONCE,
        onGetPendingJobExecutionsAccepted,
        onGetPendingJobExecutionsAcceptedSubAck);

    auto onGetPendingJobExecutionsRejected = [&](RejectedError *rejectedErr, int ioErr)
    {
        if (ioErr != AWS_OP_SUCCESS)
        {
            AGENT_LOG_WARN("Error on getting job: %s.", ErrorDebugString(ioErr));
        }
        AGENT_LOG_WARN("Getting job failed with message %s.",
                       rejectedErr->Message->c_str());
        Vector<JobExecutionSummary> dummy;
        getQueuedJobsPromise_.set_value(dummy);
        getInProgressJobsPromise_.set_value(dummy);
    };

    auto onGetPendingJobExecutionsRejectedSubAck = [&](int ioErr)
    {
        if (ioErr != AWS_OP_SUCCESS)
        {
            AGENT_LOG_WARN("Error subscribing to get Job rejected: %s", ErrorDebugString(ioErr));
            return;
        }
        subscribedRejectedPromise.set_value();
    };

    ret &= jobClient_->SubscribeToGetPendingJobExecutionsRejected(
        getPendingJobExecutionsSubscriptionRequest,
        AWS_MQTT_QOS_AT_LEAST_ONCE,
        onGetPendingJobExecutionsRejected,
        onGetPendingJobExecutionsRejectedSubAck);

    subscribedAcceptedPromise.get_future().wait();
    subscribedRejectedPromise.get_future().wait();

    return ret;
}

bool JobHandler::SubscribeDescribeJobExecution(void)
{
    bool ret = true;
    std::promise<void> subscribedAcceptedPromise;
    std::promise<void> subscribedRejectedPromise;

    DescribeJobExecutionSubscriptionRequest describeJobExecutionSubscriptionRequest;
    describeJobExecutionSubscriptionRequest.ThingName = thingName_;
    describeJobExecutionSubscriptionRequest.JobId = "+";

    auto onSubscribeDescribeJobExecutionAccepted = [&](DescribeJobExecutionResponse *response, int ioErr)
    {
        if (ioErr)
        {
            AGENT_LOG_WARN("onSubscribeDescribeJobExecutionAccepted: Error %d occurred", ioErr);
            return;
        }
        jobExecutionDataPromise_.set_value(response->Execution.value());
    };

    auto onSubscribeDescribeJobExecutionAcceptedSubAck = [&](int)
    {
        subscribedAcceptedPromise.set_value();
    };

    ret &= jobClient_->SubscribeToDescribeJobExecutionAccepted(
        describeJobExecutionSubscriptionRequest,
        AWS_MQTT_QOS_AT_LEAST_ONCE,
        onSubscribeDescribeJobExecutionAccepted,
        onSubscribeDescribeJobExecutionAcceptedSubAck);

    auto onSubscribeDescribeJobExecutionRejected = [&](RejectedError *rejectedError, int ioErr)
    {
        if (ioErr)
        {
            AGENT_LOG_WARN("onSubscribeDescribeJobExecutionRejected: Error %d occurred", ioErr);
            return;
        }
        if (rejectedError)
        {
            AGENT_LOG_WARN("Service Error %d occurred", (int)rejectedError->Code.value());
            return;
        }
    };

    auto onSubscribeDescribeJobExecutionRejectedSubAck = [&](int)
    {
        subscribedRejectedPromise.set_value();
    };

    ret &= jobClient_->SubscribeToDescribeJobExecutionRejected(
        describeJobExecutionSubscriptionRequest,
        AWS_MQTT_QOS_AT_LEAST_ONCE,
        onSubscribeDescribeJobExecutionRejected,
        onSubscribeDescribeJobExecutionRejectedSubAck);

    subscribedAcceptedPromise.get_future().wait();
    subscribedRejectedPromise.get_future().wait();

    return ret;
}

bool JobHandler::GetPendingJobExecutions(
    Vector<JobExecutionSummary>& queued, Vector<JobExecutionSummary>& inProgress)
{
    bool ret = true;
    std::promise<void> publishCompletedFuture;

    GetPendingJobExecutionsRequest getPendingJobExecutionsRequest;
    UUID uuid;
    getPendingJobExecutionsRequest.ThingName = thingName_;
    getPendingJobExecutionsRequest.ClientToken = uuid.ToString();

    // intialize promise
    getQueuedJobsPromise_ = std::promise<Vector<JobExecutionSummary>>();
    getInProgressJobsPromise_ = std::promise<Vector<JobExecutionSummary>>();

    auto onGetPendingJobExecutionsRequestSubAck = [&](int ioErr)
    {
        if (ioErr != AWS_OP_SUCCESS)
        {
            AGENT_LOG_WARN("Error: publishing to get JobExecution failed: %s", ErrorDebugString(ioErr));
            return;
        }
        publishCompletedFuture.set_value();
    };

    ret &= jobClient_->PublishGetPendingJobExecutions(
        getPendingJobExecutionsRequest,
        AWS_MQTT_QOS_AT_LEAST_ONCE,
        onGetPendingJobExecutionsRequestSubAck);

    publishCompletedFuture.get_future().wait();

    queued = getQueuedJobsPromise_.get_future().get();
    inProgress = getInProgressJobsPromise_.get_future().get();

    return ret;
}

void JobHandler::InitialJobExec(void)
{
    AGENT_LOG_DEBUG("initial job execution!");

    Vector<JobExecutionSummary> inProgressJobs;
    Vector<JobExecutionSummary> queuedJobs;

    if (!GetPendingJobExecutions(queuedJobs, inProgressJobs))
    {
        AGENT_LOG_WARN("failed to initial get Job executions");
    }

    if (!inProgressJobs.empty())
    {
        RunJobsFromSummary(inProgressJobs);
    }

    if (fs::is_regular_file(swuInstallDoneFilePath_) &&
        !fs::is_regular_file(agentStartedFilePath_))
    {
        if (std::remove(swuInstallDoneFilePath_.c_str()) == 0)
        {
            AGENT_LOG_WARN("Deleted %s. This is incorrect behavior.",
                           swuInstallDoneFilePath_.c_str());
        }
        else
        {
            AGENT_LOG_WARN("%s could not be deleted. This is incorrect behavior.",
                           swuInstallDoneFilePath_.c_str());
        }
    }

    isInitialJobGet_ = false;
    std::ofstream(agentStartedFilePath_.c_str());
    if (!queuedJobs.empty()) {
        initialQueuedJobs_ = queuedJobs;
    }
}

void JobHandler::RunJobSteps(String jobId, JsonObject jobStepObj)
{
    Vector<JsonView> jobSteps = jobStepObj.View().AsArray();
    for (JsonView jobStep: jobSteps)
    {
        JsonView action = jobStep.GetJsonObject("action");
        String actionName = action.GetString("name");
        bool rejectJob = false;
        Map<String, String> statusDetails;

        if (GetJobIdExecutingSwupdate() != "" && actionName == "SWUpdate")
        {
            rejectJob = true;
            statusDetails.insert(
                std::make_pair("reason", "ファームウェアアップデートは現在実行中です。")
            );
        }
        else if (GetJobIdExecutingCommand() != "" && actionName == "Run-Command")
        {
            rejectJob = true;
            statusDetails.insert(
                std::make_pair("reason", "コマンドは現在実行中です。")
            );
        }

        if (rejectJob)
        {
            UpdateJobState(
                jobId,
                "REJECTED",
                statusDetails
            );
            AGENT_LOG_WARN("The job is a duplicate, so it is rejected.");
            return;
        }

        PowerNotificationHandler* powerNotificationHandler = PowerNotificationHandler::Singleton();
        if (! powerNotificationHandler->InhibitSleep())
        {
            AGENT_LOG_WARN("transition to sleep state may be in progress, so abort RunJobSteps().");
            return;
        }
        JobHandler::RunJob(jobId, jobStep);
        powerNotificationHandler->AllowSleep();
    }
}

bool JobHandler::GetDescribeJobExecution(
    Vector<JobExecutionSummary> jobSummaries,
    Vector<JobExecutionData>& jobExecutions)
{
    bool ret = true;
    DescribeJobExecutionRequest describeJobExecutionRequest;
    describeJobExecutionRequest.ThingName = thingName_;
    describeJobExecutionRequest.IncludeJobDocument = true;

    // intialize promise
    jobExecutionDataPromise_ = std::promise<JobExecutionData>();

    for (size_t i = 0; i < jobSummaries.size(); ++i)
    {
        if (jobSummaries[i].JobId.has_value())
        {
            AGENT_LOG_DEBUG("job ID: %s",
                            jobSummaries[i].JobId.value().c_str());

            UUID uuid;
            describeJobExecutionRequest.ClientToken = uuid.ToString();
            describeJobExecutionRequest.JobId = jobSummaries[i].JobId.value();

            std::promise<void> publishDescribeJobExeCompletedPromise;

            auto publishDescribeJobExecutionHandler = [&](int ioErr)
            {
                if (ioErr)
                {
                    AGENT_LOG_WARN("publishDescribeJobExecutionHandler: Error %d occurred", ioErr);
                }
                publishDescribeJobExeCompletedPromise.set_value();
            };

            ret &= jobClient_->PublishDescribeJobExecution(
                std::move(describeJobExecutionRequest),
                AWS_MQTT_QOS_AT_LEAST_ONCE,
                publishDescribeJobExecutionHandler);
            publishDescribeJobExeCompletedPromise.get_future().wait();

            jobExecutions.push_back(jobExecutionDataPromise_.get_future().get());
            jobExecutionDataPromise_ = std::promise<JobExecutionData>();
        }
        else
        {
            AGENT_LOG_WARN("job does not have ID");
        }
    }
    return ret;
}

void JobHandler::RunJobsFromSummary(Vector<JobExecutionSummary> jobSummaries)
{
    Vector<JobExecutionData> jobs;
    if (!GetDescribeJobExecution(jobSummaries, jobs))
    {
        AGENT_LOG_WARN("failed to get job executions");
        return;
    }
    for (JobExecutionData job: jobs)
    {
        String jobId = job.JobId->c_str();
        JsonObject jobStepObj = job.JobDocument->View().GetJsonObjectCopy("steps");
        std::thread runJobThread(
            &JobHandler::RunJobSteps, this, jobId, jobStepObj);
        if (isInitialJobGet_)
        {
            runJobThread.join();
        }
        else
        {
            runJobThread.detach();
        }
    }
}

std::optional<RejectedError> JobHandler::UpdateJobState(
    String jobId,
    String jobStatus,
    Map<String, String> statusDetails)
{
    std::lock_guard<std::mutex> lock(mtxUpdateJobState_);
    AGENT_LOG_DEBUG("Publishing request to update job status of %s to %s...", jobId.c_str(), jobStatus.c_str());

    UpdateJobExecutionRequest request;
    UUID uuid;
    request.ClientToken = uuid.ToString();
    request.ThingName = thingName_;
    request.JobId = jobId;
    request.Status = JobStatusMarshaller::FromString(jobStatus);
    if (statusDetails.size() > 0)
        request.StatusDetails = statusDetails;

    auto publishCompleted = [jobId, jobStatus](int ioErr) {
        if (ioErr)
        {
            AGENT_LOG_WARN("Failed to update the status of %s. error: %s", jobId.c_str(), ErrorDebugString(ioErr));
            return;
        }

        AGENT_LOG_DEBUG("Succeeded to update the status of %s to %s.", jobId.c_str(), jobStatus.c_str());
    };

    jobClient_->PublishUpdateJobExecution(
        request,
        AWS_MQTT_QOS_AT_LEAST_ONCE,
        std::move(publishCompleted));
    AGENT_LOG_DEBUG("Published request to update job.");
    std::optional<RejectedError> rejectedError = finishUpdateJobStatePromise_.get_future().get();
    finishUpdateJobStatePromise_ = std::promise<std::optional<RejectedError>>();
    return rejectedError;
}

void JobHandler::RunJob(String jobId, JsonView jobStep)
{
    JsonView action = jobStep.GetJsonObject("action");
    String actionName = action.GetString("name");
    JsonView actionInput = action.GetJsonObject("input");
    Map<String, String> statusDetails;
    AGENT_LOG_DEBUG("\tactionName: %s", actionName.c_str());

    if (actionName == "Run-Command")
    {
        if (!isCommandExecAllowed_)
        {
            statusDetails.insert(
                std::make_pair("reason", "このデバイスに対するコマンド実行は許可されていません。")
            );
            UpdateJobState(jobId, "FAILED", statusDetails);
            return;
        }

        if (isInitialJobGet_)
        {
            statusDetails.insert(
                std::make_pair("reason", "コマンド実行中に予期せぬ再起動が発生したかもしれません。")
            );
            UpdateJobState(jobId, "FAILED", statusDetails);
            return;
        }

        if (GetJobIdExecutingCommand() != jobId)
        {
            AGENT_LOG_DEBUG("Run-Command!");
            SetJobIdExecutingCommand(jobId);
            SubscribeCanceledJob(jobId);
            RunCommand(jobId, actionInput);
            UnsubscribeCanceledJob(jobId);
            SetJobIdExecutingCommand("");
        }
        else
        {
            AGENT_LOG_DEBUG("the Run-Command job is running now (jobId: %s)", jobId.c_str());
        }
    }
    else if (actionName == "SWUpdate")
    {
        if (isInitialJobGet_ && fs::is_regular_file(agentStartedFilePath_))
        {
            AGENT_LOG_DEBUG("This job will not run now, the agent may have restarted without rebooting Armadillo.");
            return;
        }

        if (fs::is_regular_file(swuWaitingRebootFilePath_))
        {
            AGENT_LOG_DEBUG("Waiting for reboot after last firmware update.");
            statusDetails.insert(
                std::make_pair("reason", "前回のファームウェアアップデート後の再起動待ちです。")
            );
            UpdateJobState(jobId, "FAILED", statusDetails);
            return;
        }

        if (GetJobIdExecutingSwupdate() != jobId)
        {
            AGENT_LOG_DEBUG("SWUpdate!");
            SetJobIdExecutingSwupdate(jobId);
            ExecSWUpdate(jobId, actionInput);
        }
        else
        {
            AGENT_LOG_DEBUG("the SWUpdate job is running now (jobId: %s)", jobId.c_str());
        }
    }
    else
    {
        AGENT_LOG_WARN("Unknown action");
        statusDetails.insert(
            std::make_pair("reason", "未定義のアクションです。: " + actionName)
        );
        UpdateJobState(jobId, "FAILED", statusDetails);
        return;
    }
}

void JobHandler::RunCommand(String jobId, JsonView actionInput)
{
    Map<String, String> statusDetails;
    String jobStatus = "IN_PROGRESS";
    int timeout = -1;

    // Checks for necessary parameters before executing commands
    // If the parameter is invalid, send FAILED to twin and exit the function
    if (!actionInput.ValueExists("command"))
    {
        AGENT_LOG_WARN("The command isn't found in run-command job document.");
        jobStatus = "FAILED";
        statusDetails.insert(
            std::make_pair("reason", "Jobドキュメント内にcommandが含まれていません。")
        );
    }

    if (!actionInput.ValueExists("timeout"))
    {
        AGENT_LOG_WARN("The timeout isn't found in run-command job document.");
        jobStatus = "FAILED";
        statusDetails.insert(
            std::make_pair("reason", "Jobドキュメント内にtimeoutが含まれていません。")
        );
    }
    else
    {
        timeout = stoi_safe(actionInput.GetString("timeout"));
        if (timeout < 0 || timeout > INT_MAX / 1000) {
            AGENT_LOG_WARN("Timeout could not be converted to int or out or range: %s",
                           actionInput.GetString("timeout").c_str());
            jobStatus = "FAILED";
            statusDetails.insert(
                std::make_pair("reason", "Jobドキュメント内のtimeoutの値が不正です。")
            );
        }
    }

    UpdateJobState(jobId, jobStatus, statusDetails);
    if (jobStatus == "FAILED")
    {
        return;
    }

    // Start command execution
    String out, err;
    int exitCode;
    String command = actionInput.GetString("command");
    bool is_timeout = false;

    AGENT_LOG_DEBUG("Starting run command: %s", command.c_str());

    exitCode = executeCommand(command, out, err, timeout * 1000, &is_timeout, jobId);

    statusDetails.insert(std::make_pair("exitCode", std::to_string(exitCode).c_str()));

    bool is_truncated = false;
    bool removed_ctrl = false;
    if (!out.empty())
    {
        removed_ctrl |= removeControlChar(out);
        is_truncated |= truncateUTF8Str(out, 1024);
        statusDetails.insert(std::make_pair("stdout", out));
    }
    if (!err.empty())
    {
        removed_ctrl |= removeControlChar(err);
        is_truncated |= truncateUTF8Str(err, 1024);
        statusDetails.insert(std::make_pair("stderr", err));
    }
    if (removed_ctrl)
    {
        AGENT_LOG_WARN("statusDetail was removed Ctrl Character");
        statusDetails.insert(std::make_pair("removed_ctrl", "1"));
    }
    if (is_truncated)
    {
        AGENT_LOG_WARN(
            "output of command \"%s\" was truncated because it exceeded 1024 characters in UTF-8",
            command.c_str());
        statusDetails.insert(std::make_pair("is_truncated", "1"));
    }

    if (is_timeout)
    {
        statusDetails.insert(std::make_pair("timeout", "1"));
    }
    else
    {
        statusDetails.insert(std::make_pair("timeout", "0"));
    }

    if (exitCode != 0)
    {
        jobStatus = "FAILED";
    }
    else
    {
        jobStatus = "SUCCEEDED";
    }

    std::optional<RejectedError> rejectedError = UpdateJobState(jobId, jobStatus, statusDetails);
    if (rejectedError)
    {
        statusDetails.erase("stdout");
        statusDetails.erase("stderr");
        statusDetails.insert(
            std::make_pair(
                "rejected_err_code",
                std::to_string(static_cast<int>(rejectedError->Code.value()))));
        UpdateJobState(jobId, jobStatus, statusDetails);
    }

    AGENT_LOG_DEBUG("Done running command.");
}

bool JobHandler::CheckInstallDoneFile(String jobId)
{
    if (!fs::is_regular_file(swuInstallDoneFilePath_))
    {
        AGENT_LOG_WARN("file not found: %s", swuInstallDoneFilePath_.c_str());
        return false;
    }

    String fileContent = readFileContents(swuInstallDoneFilePath_);
    if (fileContent.empty())
    {
        AGENT_LOG_WARN("can not read contents of %s", swuInstallDoneFilePath_.c_str());
        return false;
    }
    while (fileContent.back() == '\n')
    {
        fileContent.pop_back();
    }

    AGENT_LOG_DEBUG("fileContent: %s", fileContent.c_str());
    if (fileContent != jobId.c_str())
    {
        AGENT_LOG_WARN("contents of the retrieved jobid and the stored jobid are different");
        return false;
    }
    return true;
}

void JobHandler::ExecSWUpdate(String jobId, JsonView actionInput)
{
    String out, err;
    int exitCode;
    String jobStatus = "IN_PROGRESS";
    Map<String, String> statusDetails;

    if (CheckInstallDoneFile(jobId))
    {
        executeCommand("abos-ctrl status --rollback -q", out, err);

        if (!err.empty())
        {
            jobStatus = "FAILED";
            statusDetails.insert(std::make_pair("errMsg", "ファームウェアがロールバックされました。"));
        }
        else
        {
            jobStatus = "SUCCEEDED";

            out = readFileContents("/etc/sw-versions");
            if (!out.empty())
            {
                statusDetails.insert(std::make_pair("afterVer", out));
            }

            // set beforeVer to statusDetails
            String rootPartition;
            executeCommand("basename $(swupdate -g)", rootPartition, err);
            // remove trailing newline code
            while (rootPartition.back() == '\n')
                rootPartition.pop_back();

            std::optional<char> partNum;
            if (!rootPartition.empty())
            {
                partNum = rootPartition.back();
                rootPartition.pop_back();
            }

            if (partNum.has_value() && std::isdigit(partNum.value()))
            {
                String beforeVerFilePath = "/var/log/swupdate/sw-versions-" + rootPartition;
                switch (partNum.value())
                {
                case '1':
                    statusDetails.insert(
                        std::make_pair(
                            "beforeVer",
                            readFileContents((beforeVerFilePath + "2").c_str())));
                    break;
                case '2':
                    statusDetails.insert(
                        std::make_pair(
                            "beforeVer",
                            readFileContents((beforeVerFilePath + "1").c_str())));
                    break;
                }
            }
        }

        UpdateJobState(jobId, jobStatus, statusDetails);

        AGENT_LOG_DEBUG("Done executing SWUpdate.");

        if (std::remove(swuInstallDoneFilePath_.c_str()) != 0)
        {
            AGENT_LOG_WARN("%s could not be deleted. This is incorrect behavior.",
                           swuInstallDoneFilePath_.c_str());
        }

        SetJobIdExecutingSwupdate("");
    }
    else
    {
        if (isInitialJobGet_)
        {
            jobStatus = "FAILED";
            statusDetails.insert(
                std::make_pair("reason", "不正なSWUpdateです。")
            );
        }
        else if (!actionInput.ValueExists("url"))
        {
            AGENT_LOG_WARN("The url isn't found in swupdate job document.");
            jobStatus = "FAILED";
            statusDetails.insert(
                std::make_pair("reason", "Jobドキュメント内にS3のURLが含まれていません。")
            );
        }
        else if (!actionInput.ValueExists("postAction"))
        {
            AGENT_LOG_WARN("The postAction isn't found in swupdate job document.");
            jobStatus = "FAILED";
            statusDetails.insert(
                std::make_pair("reason", "Jobドキュメント内にpostActionが含まれていません。")
            );
        }

        if (jobStatus == "FAILED")
        {
            UpdateJobState(jobId, jobStatus, statusDetails);
            SetJobIdExecutingSwupdate("");
            return;
        }

        String url = actionInput.GetString("url");
        String postAction = actionInput.GetString("postAction");

        if (postAction != "reboot" && postAction != "container" && postAction != "wait")
        {
            AGENT_LOG_WARN("unknown postAction: %s", postAction.c_str());
            statusDetails.insert(
                std::make_pair("reason", "未定義のpostActionです。: " + postAction)
            );
            jobStatus = "FAILED";
            UpdateJobState(jobId, jobStatus, statusDetails);
            SetJobIdExecutingSwupdate("");
            return;
        }

        const char* download_speed_option = "";
        if (executeCommand(
                "/usr/libexec/armadillo-twin/is_lte_catm1_default", out, err
            ) == 0)
        {
            download_speed_option = " -n 12k";
        }
        String command = "env SWUPDATE_ARMADILLO_TWIN=" + jobId
            + " SW_ALLOW_ROLLBACK=1 MKSWU_POST_ACTION=" + postAction
            + " swupdate -d '-u " + url + download_speed_option + "'";

        String swVersion = readFileContents("/etc/sw-versions");
        statusDetails.insert(std::make_pair("beforeVer", swVersion));
        statusDetails.insert(std::make_pair("waitingReboot", "0"));

        UpdateJobState(jobId, jobStatus, statusDetails);

        AGENT_LOG_DEBUG("Starting executing SWUpdate.");

        exitCode = executeCommand(command, out, err);

        if (exitCode == 0 && !fs::is_regular_file(swuInstallDoneFilePath_))
        {
            // postAction == container
            // (twin agent "done" file is not created if not rebooting)
            jobStatus = "SUCCEEDED";

            swVersion = readFileContents("/etc/sw-versions");
            statusDetails.insert(std::make_pair("afterVer", swVersion));
            SetJobIdExecutingSwupdate("");
        }
        else if (fs::is_regular_file(swuInstallDoneFilePath_))
        {
            // older versions of swupdate can succeed with SIGKILL status,
            // so we did not check exitCode on purpose
            if (fs::is_regular_file(swuWaitingRebootFilePath_))
            {
                // postAction == wait
                jobStatus = "IN_PROGRESS";
                statusDetails["waitingReboot"] = "1";
            }
            else
            {
                // postAction == reboot or container(with OS updates)
                AGENT_LOG_DEBUG("auto reboot by SWUpdate.");
                return;
            }
        }
        else
        {
            // SWUpdate failed
            AGENT_LOG_WARN("failed to execute SWUpdate.");
            jobStatus = "FAILED";
            statusDetails.insert(std::make_pair("errMsg", err));
            SetJobIdExecutingSwupdate("");
        }

        UpdateJobState(jobId, jobStatus, statusDetails);

        AGENT_LOG_DEBUG("Done executing SWUpdate.");
    }
}
