/**
 * Copyright (C) 2023-2024 Atmark Techno, Inc. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */

#include <stdio.h>

#include "dbus.h"
#include "agent_log.h"

DBusConnection *dbusConnect(void)
{
    DBusConnection *connection;
    DBusError error;
    DBusBusType type = DBUS_BUS_SYSTEM;
    dbus_error_init(&error);
    connection = dbus_bus_get(type, &error);
    if (!connection)
    {
        AGENT_LOG_WARN("Failed to open connection to system message bus: %s",
                error.message);
        dbus_error_free(&error);
        return NULL;
    }
    return connection;
}

void dbusDisconnect(DBusConnection *dbus)
{
    dbus_connection_unref(dbus);
}

int modemGetSignalQuality(DBusConnection *connection)
{
    const char * const path = "/org/freedesktop/ModemManager1/Modem/0";
    const char * const dest = "org.freedesktop.ModemManager1";
    const char * const name = "org.freedesktop.DBus.Properties";
    const char * const member = "Get";

    DBusMessage *message;
    message = dbus_message_new_method_call(NULL, path, name, member);
    if (!message || !dbus_message_set_destination(message, dest)) {
        // ENOMEM
        return -1;
    }

    // append args
    DBusMessageIter arg_iter;
    dbus_message_iter_init_append(message, &arg_iter);
    const char * const arg1 = "org.freedesktop.ModemManager1.Modem";
    const char * const arg2 = "SignalQuality";
    dbus_message_iter_append_basic(&arg_iter, DBUS_TYPE_STRING, &arg1);
    dbus_message_iter_append_basic(&arg_iter, DBUS_TYPE_STRING, &arg2);

    // send and get reply
    DBusMessage *reply;
    DBusError error;
    dbus_error_init(&error);
    reply = dbus_connection_send_with_reply_and_block(connection, message,
                                                      10 /* timeout */, &error);
    if (dbus_error_is_set(&error) || !reply) {
        // if (dbus_error_is_set(&error))
        //     AGENT_LOG_WARN("Error %s: %s", error.name, error.message);
        dbus_error_free(&error);
        return -1;
    }

    // signal quality reply:
    // variant       struct {
    //         uint32 80
    //         boolean true
    //      }

    DBusMessageIter variant_iter;
    dbus_message_iter_init(reply, &variant_iter);
    int type = dbus_message_iter_get_arg_type(&variant_iter);
    if (type != DBUS_TYPE_VARIANT)
    {
        AGENT_LOG_WARN("Signal quality reply was not a variant: %d", type);
        return -1;
    }

    DBusMessageIter struct_iter;
    dbus_message_iter_recurse(&variant_iter, &struct_iter);
    type = dbus_message_iter_get_arg_type(&struct_iter);
    if (type != DBUS_TYPE_STRUCT)
    {
        AGENT_LOG_WARN("Signal quality reply variant was not a struct: %d", type);
        return -1;
    }

    DBusMessageIter value_iter;
    dbus_message_iter_recurse(&struct_iter, &value_iter);
    type = dbus_message_iter_get_arg_type(&value_iter);
    if (type != DBUS_TYPE_UINT32)
    {
        AGENT_LOG_WARN("Signal quality reply struct did not contain a uint32: %d", type);
        return -1;
    }

    dbus_int32_t val;
    dbus_message_iter_get_basic(&value_iter, &val);
    if (val > 100)
    {
        AGENT_LOG_WARN("Signal quality value seems bogus: %d", val);
        return -1;
    }

    dbus_message_unref(reply);
    dbus_message_unref(message);

    return val;
}

String modemGetLocation(DBusConnection *connection)
{
    const char * const path = "/org/freedesktop/ModemManager1/Modem/0";
    const char * const dest = "org.freedesktop.ModemManager1";
    const char * const name = "org.freedesktop.ModemManager1.Modem.Location";
    const char * const member = "GetLocation";

    DBusMessage *message;
    message = dbus_message_new_method_call(NULL, path, name, member);
    if (!message || !dbus_message_set_destination(message, dest)) {
        // ENOMEM
        return String("");
    }

    // send and get reply
    DBusMessage *reply;
    DBusError error;
    dbus_error_init(&error);
    reply = dbus_connection_send_with_reply_and_block(connection, message,
                                                      10 /* timeout */, &error);
    if (dbus_error_is_set(&error) || !reply) {
        // if (dbus_error_is_set(&error))
        //     AGENT_LOG_WARN("Error %s: %s", error.name, error.message);
        dbus_error_free(&error);
        return String("");
    }

    if (!reply) {
        AGENT_LOG_WARN("No error but no reply either?");
        return String("");
    }

    // location reply:
    //    array [
    //      dict entry(
    //         uint32 1
    //         variant             string "440,10,0,BC25712,30BE"
    //      )
    //   ]

    DBusMessageIter array_iter;
    dbus_message_iter_init(reply, &array_iter);
    int type = dbus_message_iter_get_arg_type(&array_iter);
    if (type != DBUS_TYPE_ARRAY)
    {
        AGENT_LOG_WARN("Location reply was not an array: %d", type);
        return String("");
    }

    DBusMessageIter dict_iter;
    dbus_message_iter_recurse(&array_iter, &dict_iter);
    type = dbus_message_iter_get_arg_type(&dict_iter);
    if (type != DBUS_TYPE_DICT_ENTRY)
    {
        AGENT_LOG_WARN("Location reply array did not contain dicts: %d", type);
        return String("");
    }

    DBusMessageIter variant_iter;
    dbus_message_iter_recurse(&dict_iter, &variant_iter);
    // skip '1' key of dict
    dbus_message_iter_next(&variant_iter);
    type = dbus_message_iter_get_arg_type(&variant_iter);
    if (type != DBUS_TYPE_VARIANT)
    {
        AGENT_LOG_WARN("Location reply dict value was not a variant: %d", type);
        return String("");
    }

    DBusMessageIter value_iter;
    dbus_message_iter_recurse(&variant_iter, &value_iter);
    type = dbus_message_iter_get_arg_type(&value_iter);
    if (type != DBUS_TYPE_STRING)
    {
        AGENT_LOG_WARN("Location reply variant was not a string: %d", type);
        return String("");
    }

    char *loc;
    dbus_message_iter_get_basic(&value_iter, &loc);
    String location = String(loc);

    dbus_message_unref(reply);
    dbus_message_unref(message);

    return location;
}
