/**
 * Copyright (C) 2023-2024 Atmark Techno, Inc. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */

#include <sys/statvfs.h>
#include <mntent.h>
#include <unistd.h>
#include <signal.h>
#include <regex>

#include "agent_utils.h"
#include "agent_log.h"

String readFileContents(const String& filePath)
{
    std::ifstream ifs(filePath.c_str());

    if (!ifs) {
        AGENT_LOG_DEBUG("Failed to open file: %s", filePath.c_str());
        return "";
    }

    String content{std::istreambuf_iterator<char>{ifs}, {}};
    return content;
}

// convert string to integer or return INT_MIN + set errno on error
// (including not consuming the whole string)
int stoi_safe(const String& str)
{
    long int val;
    char *endp;
    errno = 0;
    val = strtol(str.c_str(), &endp, 10);
    if (endp[0] != '\0' && endp[0] != '\n')
    {
        errno = EINVAL;
        return INT_MIN;
    }
    if (val < INT_MIN || val > INT_MAX)
    {
        errno = ERANGE;
        return INT_MIN;
    }
    return val;
}

int get_nproc(void)
{
    cpu_set_t *set = CPU_ALLOC(1024);

    if (sched_getaffinity(0, CPU_ALLOC_SIZE(1024), set) != 0) {
        AGENT_LOG_WARN("sched_getaffinity failed: %d", errno);
        return -1;
    }
    int count = CPU_COUNT(set);
    CPU_FREE(set);
    return count;
}

int64_t statvfs_free_kb(const char *path)
{
    struct statvfs buf;
    int rc;

    rc = statvfs(path, &buf);
    if (rc < 0) {
        AGENT_LOG_WARN("statvfs %s failed: %d", path, errno);
        return -1;
    }
    static_assert(sizeof(buf.f_bavail) >= 8, "statvfs' f_bavail should be at least 64bit");
    if (buf.f_bavail > INT64_MAX / buf.f_bsize) {
        // We could assert f_bsize is a multiple of 1024 and relax
        // the check by a factor of 1k, but in practice this should be
        // ok for a while... 
        AGENT_LOG_WARN("more than INT64_MAX bytes available in %s!", path);
        return -1;
    }
    return (buf.f_bavail * buf.f_bsize) / 1024;
}

String getMountSource(const String& mountPoint)
{
    FILE* mounts = setmntent("/proc/mounts", "r");
    struct mntent* entry;

    while ((entry = getmntent(mounts)) != nullptr) {
        if (mountPoint == String(entry->mnt_dir)) {
            endmntent(mounts);
            return entry->mnt_fsname;
        }
    }

    endmntent(mounts);
    // if the requested mount point is not found, return empty
    AGENT_LOG_WARN("could not find the mount source of %s", mountPoint.c_str());
    return "";
}

bool createPidFile(int pid, const String& pidFilePath)
{
    std::ofstream outputFile(pidFilePath.c_str());
    if (!outputFile.is_open())
    {
        AGENT_LOG_WARN("failed to open %s", pidFilePath.c_str());
        return false;
    }
    outputFile << pid << std::endl;
    outputFile.close();
    return true;
}

bool checkAndCreatePidFile(void)
{
    String pidFilePath = "/run/armadillo-twin-agentd.pid";
    int myPid = getpid();

    String pidFile = readFileContents(pidFilePath);
    if (pidFile.empty())
        return createPidFile(myPid, pidFilePath);

    int runningPid = stoi_safe(pidFile);
    if (runningPid <= 1)
    {
        AGENT_LOG_WARN("failed to stoi_safe(%s)", pidFile.c_str());
        return false;
    }
    int result = kill(runningPid, 0);
    if (result == 0)
    {
        AGENT_LOG_WARN("armadillo-twin-agent is already running.");
        return false;
    }
    else
    {
        return createPidFile(myPid, pidFilePath);
    }
}

std::size_t getCharSizeUTF8(const char c)
{
    /*
     * In UTF-8, the number of bytes in a character can be determined as follows.
     *  - The first 1 bit is 0 -> 1 byte
     *  - The first 3 bits are 110 -> 2 bytes
     *  - The first 4 bits are 1110 -> 3 bytes
     *  - The first 5 bits are 11110 -> 4 bytes
     */
    if ((c & 0x80) == 0)
        return 1;
    if ((c & 0xe0) == 0xc0)
        return 2;
    if ((c & 0xf0) == 0xe0)
        return 3;
    return 4;
}

bool truncateUTF8Str(String& str, int length)
{
    int l = 0;
    std::size_t index = 0;
    while (index < str.length())
    {
        l++;
        if (l > length)
        {
            str.resize(index);
            // Returns true if string truncation occurs
            return true;
        }
        std::size_t size = getCharSizeUTF8(str[index]);
        index += size;
    }
    // Return false if no string truncation occurs
    return false;
}

bool removeControlChar(String& str)
{
    // It is assumed that strings do not contain unexpected
    // (non control characters) 'x1b', but UTF-8 always satisfies this assumption
    std::regex pattern("\x1b\\[[\x30-\x3f]*[\x20-\x2f]*[\x40-\x7e]");
    String result = std::regex_replace(str, pattern, "");

    if (result != str)
    {
        str = result;
        return true;
    }
    return false;
}
