/**
 * Copyright (C) 2023-2024 Atmark Techno, Inc. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
#pragma once

#include <chrono>
#include <sys/time.h>

using Monotonic = std::chrono::steady_clock;
using monotonic_point = std::chrono::time_point<Monotonic, std::chrono::nanoseconds>;
using monotonic_float_point = std::chrono::time_point<Monotonic, std::chrono::duration<float>>;

static inline long long monotonic_us_diff(monotonic_point &a, monotonic_point &b) {
    return std::chrono::duration_cast<std::chrono::microseconds>(a - b).count();
}

static inline float monotonic_float_diff(monotonic_point &a, monotonic_point &b) {
    std::chrono::duration<float> diff = a - b;
    return diff.count();
}
static inline float monotonic_float_since_epoch(monotonic_point &time) {
    monotonic_float_point time_float = time;
    return time_float.time_since_epoch().count();
}

static inline struct timeval *time_left(
        monotonic_point &start_time,
        int timeout_ms,
        struct timeval *tv_left)
{
    if (timeout_ms <= 0)
        return NULL;

    auto now = Monotonic::now();
    // compute in us (for timeval)
    long long usec_left = timeout_ms * 1000 - monotonic_us_diff(now, start_time);
    if (usec_left < 0)
        usec_left = 0;
    tv_left->tv_sec = (long int)usec_left / 1000000;
    tv_left->tv_usec = (long int)usec_left % 1000000;
    return tv_left;
}
