/**
 * Copyright (C) 2023-2024 Atmark Techno, Inc. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
#include <cstdarg>
#include <fstream>
#include <iomanip>
#include <filesystem>

#include "agent_log.h"

namespace fs = std::filesystem;

AgentLogger& AgentLogger::instance(void)
{
    static AgentLogger instance;
    return instance;
}

void AgentLogger::setConsoleLogLevel(String level)
{
    if (level.empty())
        return;

    if(!stringToLogLevel(level, consoleLogLevel_))
        AGENT_LOG_WARN("Console log level was not changed.");
}

void AgentLogger::setLogFileLogLevel(String level)
{
    if (level.empty())
        return;

    if(!stringToLogLevel(level, logFileLogLevel_))
        AGENT_LOG_WARN("Logfile log level was not changed.");
}

void AgentLogger::log(AgentLogLevel level, const char* format, ...)
{
    std::lock_guard<std::mutex> lock(mtx_);

    logRotate(AGENT_LOG_FILEPATH, MAX_LOG_SIZE);

    bool output_console = (level >= consoleLogLevel_);
    bool output_logfile = (level >= logFileLogLevel_);

    if (!output_console && !output_logfile)
        return;

    va_list args;
    if (output_console)
    {
        va_start(args, format);
        vfprintf(stdout, format, args);
        fprintf(stdout, "\n");
        va_end(args);
    }

    if (output_logfile)
    {
        String prefix = getCurrentTime() + " [" + logLevelToString(level) + "] ";
        FILE *fp = fopen(AGENT_LOG_FILEPATH, "a");
        if(fp == NULL)
        {
            fprintf(stderr, "failed to open %s\n", AGENT_LOG_FILEPATH);
            return;
        }

        va_start(args, format);
        fprintf(fp, "%s", prefix.c_str());
        vfprintf(fp, format, args);
        fprintf(fp, "\n");
        va_end(args);
        fclose(fp);
    }
}

String AgentLogger::logLevelToString(AgentLogLevel level)
{
    switch (level)
    {
    case AgentLogLevel::DEBUG:
        return "DEBUG";
    case AgentLogLevel::INFO:
        return "INFO";
    case AgentLogLevel::WARN:
        return "WARN";
    case AgentLogLevel::ERROR:
        return "ERROR";
    case AgentLogLevel::NONE:
    default:
        return "";
    }
}

bool AgentLogger::stringToLogLevel(const String& strLevel, AgentLogLevel& agentLogLevel)
{
    auto it = levelStringMap.find(strLevel);
    if (it != levelStringMap.end())
    {
        agentLogLevel = it->second;
        return true;
    }
    else
    {
        AGENT_LOG_WARN("unknown log level: %s", strLevel.c_str());
        return false;
    }
}

String AgentLogger::getCurrentTime(void)
{
    auto now = std::chrono::system_clock::to_time_t(std::chrono::system_clock::now());
    StringStream ss;
    ss << std::put_time(std::localtime(&now), "%FT%T%z");
    return ss.str();
}

void AgentLogger::logRotate(const String& logFilePath, const size_t logFileMaxSize)
{
    fs::path currentPath(logFilePath);
    if (!fs::exists(currentPath) || fs::file_size(currentPath) < logFileMaxSize)
        return;

    for (int i = MAX_LOG_GENERATION; i > 1; i--)
    {
        fs::path oldFilePath = logFilePath + "." + String(std::to_string(i - 1));
        fs::path newFilePath = logFilePath + "." + String(std::to_string(i));
        if (!fs::exists(oldFilePath))
            continue;
        fs::rename(oldFilePath, newFilePath);
    }

    fs::path nextGenPath = logFilePath + ".1";
    fs::rename(currentPath, nextGenPath);
}
