/**
 * Copyright (C) 2023-2024 Atmark Techno, Inc. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
#include <iostream>
#include <unistd.h>

#include <aws/crt/Api.h>

#include "../agent_utils.h"
#include "../shadow_factory.h"

using namespace Aws::Crt;

int check_cpu_usage(JsonView *hb) {
    // cpu_{idle+user+system+interrupt+other} should be roughly the same as
    // number of cores
    int cpu_total = hb->GetInteger("idle")
        + hb->GetInteger("user")
        + hb->GetInteger("system")
        + hb->GetInteger("interrupt")
        + hb->GetInteger("other");
    if (cpu_total < 75 || cpu_total > 125) {
        fprintf(stderr, "cpu total is weird: had %d, expected 75-125\n",
                cpu_total);
        return 1;
    }

    return 0;
}

int main() {
    int rc = 0;
    // init aws api
    ApiHandle apiHandle;

    // the automated test only checks there's no segfault or similar,
    // run manually and inspect output to validate results...
    ShadowFactory* shadow_factory = new ShadowFactory("atmark-techno.com");
    Shadow *device_info_shadow = shadow_factory->CreateShadow(ShadowPurpose::DeviceInfo);
    if (!device_info_shadow) {
        printf("could not create device info shadow\n");
        return 1;
    }
    JsonObject di_json = device_info_shadow->get();
    std::cout << di_json.View().WriteReadable() << std::endl;
    delete device_info_shadow;

    device_info_shadow = shadow_factory->CreateShadow(
        ShadowPurpose::DeviceInfo, true);
    if (!device_info_shadow) {
        printf("could not create device info shadow (isCommandexecallowed=true)\n");
        return 1;
    }
    di_json = device_info_shadow->get();
    std::cout << di_json.View().WriteReadable() << std::endl;

    Shadow *heart_beat_shadow = shadow_factory->CreateShadow(ShadowPurpose::HeartBeat);
    if (!heart_beat_shadow) {
        printf("could not create heart beat shadow\n");
        return 1;
    }
    JsonObject hb_json = heart_beat_shadow->get();
    std::cout << hb_json.View().WriteReadable() << std::endl;

    // sanity checks on received values
    JsonView hb_view = hb_json.View().GetJsonObject("heart_beat").GetJsonObject("cpu");
    if (check_cpu_usage(&hb_view)) rc = 1;

    // wait a bit and get again to check updates
    usleep(10000);
    hb_json = heart_beat_shadow->get();
    std::cout << hb_json.View().WriteReadable() << std::endl;

    // ... same sanity checks
    hb_view = hb_json.View().GetJsonObject("heart_beat").GetJsonObject("cpu");
    if (check_cpu_usage(&hb_view)) rc = 1;

    return rc;
}
