/**
 * Copyright (C) 2023-2024 Atmark Techno, Inc. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
#include <iostream>
#include <string>

#include "../execute_command.h"

// kill timeout is slow so only test if requested. Uncomment to test
//#define TEST_KILL_TIMEOUT

struct commandTemplate {
    String command;
    int timeout;
    int rc;
    String out, err;
    bool is_timeout;
};

std::string boolToString(bool value) {
    std::ostringstream ss;
    ss << std::boolalpha << value;
    return ss.str();
}

int testExecCommand(struct commandTemplate *test) {
    String out, err;
    bool is_timeout;

    int rc = executeCommand(test->command, out, err, test->timeout, &is_timeout);
    if (rc != test->rc) {
        fprintf(stderr, "'%s' returned %d instead of %d\n",
            test->command.c_str(), rc, test->rc);
        return 1;
    }
    if (out != test->out) {
        fprintf(stderr, "'%s' stdout was '%s' instead of '%s'\n",
            test->command.c_str(), out.c_str(), test->out.c_str());
        return 1;
    }
    if (err != test->err) {
        fprintf(stderr, "'%s' stderr was '%s' instead of '%s'\n",
            test->command.c_str(), err.c_str(), test->err.c_str());
        return 1;
    }
    if (is_timeout != test->is_timeout) {
        fprintf(stderr, "'%s' is_timeout was '%s' instead of '%s'\n",
            test->command.c_str(),
            boolToString(is_timeout).c_str(),
            boolToString(test->is_timeout).c_str());
        return 1;
    }
    return 0;
}

int main() {
    struct commandTemplate tests[] = {
        { "printf \"%s\\n\" \"a b\" c", 0, 0, "a b\nc\n", "", false },
        { "false", 0, 1, "", "", false },
        { "echo stderr >&2", 0, 0, "", "stderr\n", false },
        { "echo mixOut; echo mixErr >&2", 0, 0, "mixOut\n", "mixErr\n", false },
        { "echo mixOut; sleep 0.2; echo mixErrDelay >&2", 0, 0, "mixOut\n", "mixErrDelay\n", false },
        { "echo mixOut; (sleep 0.2; echo lostErr >&2) &", 0, 0, "mixOut\n", "", false },
        { "echo start; exec sleep 10", 200, 128+SIGTERM, "start\n", "", true },
#ifdef TEST_KILL_TIMEOUT
        // (note: ctest waits for indirect child processes to end, so waits a bit
        // after test end for this sleep 5 to finish)
        { "echo start; trap 'echo gotTerm' TERM; sleep 1; echo middle; sleep 5; echo lostEnd",
            200, 128+SIGKILL, "start\ngotTerm\nmiddle\n", "", true },
#endif
    };
    int rc = 0;
    for (size_t i = 0; i < sizeof(tests)/sizeof(*tests); i++) {
        rc += testExecCommand(tests + i);
    }
    return rc;
}
