/**
 * Copyright (C) 2023-2024 Atmark Techno, Inc. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */

#include <filesystem>
#include <regex>

#include "shadow_obj.h"
#include "execute_command.h"
#include "agent_utils.h"
#include "ping.h"
#include "mmc.h"
#include "version.h"
#include "agent_log.h"

// 仮想デストラクタ
Shadow::~Shadow(void){};

void Shadow::AppendTimeStamp(JsonObject &shadow)
{
    shadow.WithInteger("timestamp", std::time(nullptr));
}

DeviceInfoShadow::DeviceInfoShadow(bool isCommandExecAllowed)
    : isCommandExecAllowed_(isCommandExecAllowed) {}

JsonObject DeviceInfoShadow::get(void)
{
    JsonObject shadow, device_info;
    AppendSWVersions(device_info);
    AppendDeviceId(device_info);
    AppendTimeStamp(device_info);
    AppendCommandExecAllowed(device_info);
    AppendAgentVersion(device_info);
    AppendBootTime(device_info);
    shadow.WithObject("device_info", device_info);
    return shadow;
}

void DeviceInfoShadow::AppendSWVersions(JsonObject &shadow)
{
    String out;
    out = readFileContents("/etc/sw-versions");
    if (out.empty())
    {
        AGENT_LOG_WARN("Failed to get Base OS version");
        return;
    }

    shadow.WithString("version", out.c_str());
}

void DeviceInfoShadow::AppendDeviceId(JsonObject &shadow)
{
    String out, err;
    int exitCode;
    exitCode = executeCommand("device-info -s", out, err);
    if (exitCode != 0)
    {
        AGENT_LOG_WARN("Failed to get Device ID: %s", err.c_str());
        return;
    }

    if (out.back() == '\n')
    {
        out.pop_back();
    }

    shadow.WithString("device_id", out.c_str());
}

void DeviceInfoShadow::AppendCommandExecAllowed(JsonObject &shadow)
{
    shadow.WithBool("allow_exec_command", isCommandExecAllowed_);
}

void DeviceInfoShadow::AppendAgentVersion(JsonObject &shadow)
{
    shadow.WithString("agent_version", VERSION);
}

void DeviceInfoShadow::AppendBootTime(JsonObject &shadow)
{
    String proc_uptime = readFileContents("/proc/uptime");
    if (proc_uptime.empty())
        return;

    std::istringstream iss(proc_uptime.c_str());
    double uptime;
    iss >> uptime;
    shadow.WithInteger(
        "boot_time",
        static_cast<int>(std::time(nullptr) - static_cast<time_t>(uptime)));
}

HeartBeatShadow::HeartBeatShadow(String pingDst) : pingDst_(pingDst)
{
    // get device file from which /var/app/volumes is mounted
    // (/dev/mmcblkXpY)
    String appMountSrc = getMountSource("/var/app/volumes");

    // delete partition number (/dev/mmcblkX) and set to bootDevice_
    std::regex deviceBasePattern("/dev/mmcblk\\d+");
    std::cmatch match;
    if (std::regex_search(appMountSrc.c_str(), match, deviceBasePattern))
    {
        bootDevice_ = match[0].str();
    }
    else
    {
        AGENT_LOG_WARN("Could not get boot device");
        bootDevice_ = "";
    }

    nproc_ = get_nproc();
    if (nproc_ < 0)
    {
        AGENT_LOG_WARN("Could not get nproc, assuming 1 core");
        nproc_ = 1;
    }
}

JsonObject HeartBeatShadow::get(void)
{
    JsonObject shadow, heart_beat;

    AppendNetPing(heart_beat);
    AppendAppAvailableSpace(heart_beat);
    AppendMemoryAvailableSpace(heart_beat);
    AppendDiskIo(heart_beat);
    AppendEmmcPreEolInfo(heart_beat);
    AppendTimeStamp(heart_beat);
    AppendContainerStatus(heart_beat);

    DBusConnection *dbus = dbusConnect();
    if (dbus)
    {
        AppendCellularSiganalQuality(heart_beat, dbus);
        AppendCellularLocation(heart_beat, dbus);
        dbusDisconnect(dbus);
    }
    // keep time-sensitive values close to lastRunTs_
    // to avoid noise from e.g. slow ping time
    AppendCPUInfo(heart_beat);

    lastRunTs_ = std::chrono::steady_clock::now();
    firstRun_ = false;

    shadow.WithObject("heart_beat", heart_beat);
    return shadow;
}

int HeartBeatShadow::CountString(String str, String sub)
{
    size_t offset = 0;
    int count = 0;

    while ((offset = str.find(sub, offset)) != String::npos)
    {
        count++;
        // move offset forward by substring length
        offset += sub.length();
    }
    return count;
}

void HeartBeatShadow::AppendCPUTemp(JsonObject &shadow)
{
    String out = readFileContents("/sys/class/thermal/thermal_zone0/temp");

    if (out.empty())
        return;

    int temp = stoi_safe(out);
    if (temp == INT_MIN)
    {
        AGENT_LOG_WARN("Invalid temp in /sys/class/thermal/thermal_zone0/temp: %s", out.c_str());
        return;
    }

    shadow.WithInteger("temp", temp);
}

void HeartBeatShadow::AppendCPUInfo(JsonObject &shadow)
{
    String out = readFileContents("/proc/stat");

    // line with 'cpu ' has aggregate for all CPUs
    auto offset = out.find("cpu ");
    if (offset == String::npos)
    {
        AGENT_LOG_WARN("No 'cpu ' in /proc/stat!");
        return;
    }

    /* from Documentation/filesystems/proc.rst
     * The meanings of the columns are as follows, from left to right:
     *
     * - user: normal processes executing in user mode
     * - nice: niced processes executing in user mode
     * - system: processes executing in kernel mode
     * - idle: twiddling thumbs
     * - iowait: In a word, iowait stands for waiting for I/O to complete. But there
     *   are several problems:
     *
     *   1. CPU will not wait for I/O to complete, iowait is the time that a task is
     *      waiting for I/O to complete. When CPU goes into idle state for
     *      outstanding task I/O, another task will be scheduled on this CPU.
     *   2. In a multi-core CPU, the task waiting for I/O to complete is not running
     *      on any CPU, so the iowait of each CPU is difficult to calculate.
     *   3. The value of iowait field in /proc/stat will decrease in certain
     *      conditions.
     *
     *   So, the iowait is not reliable by reading from /proc/stat.
     * - irq: servicing interrupts
     * - softirq: servicing softirqs
     * - steal: involuntary wait
     * - guest: running a normal guest
     * - guest_nice: running a niced guest
     */

    std::istringstream iss(out.substr(offset + 4).c_str());
    uint64_t user, nice, system, idle, iowait;
    uint64_t irq, softirq, steal, guest, guest_nice;

    iss >> user >> nice >> system >> idle >> iowait;
    iss >> irq >> softirq >> steal >> guest >> guest_nice;

    if (!iss)
    {
        AGENT_LOG_WARN("/proc/stat line was not as expected: %s",
                       out.substr(offset + 4).c_str());
        return;
    }

    // group for twin
    user = user + nice;
    uint64_t interrupt = irq + softirq;
    uint64_t other = iowait + steal + guest + guest_nice;

    auto now = Monotonic::now();
    float time_diff;
    if (firstRun_)
    {
        time_diff = monotonic_float_since_epoch(now);
    } else
    {
        time_diff = monotonic_float_diff(now, lastRunTs_);
    }

    int idle_diff = (idle - cpuIdle_) / time_diff / nproc_;
    int user_diff = (user - cpuUser_) / time_diff / nproc_;
    int system_diff = (system - cpuSystem_) / time_diff / nproc_;
    int interrupt_diff = (interrupt - cpuInterrupt_) / time_diff / nproc_;
    int other_diff = (other - cpuOther_) / time_diff / nproc_;

    cpuIdle_ = idle;
    cpuUser_ = user;
    cpuSystem_ = system;
    cpuInterrupt_ = interrupt;
    cpuOther_ = other;

    JsonObject cpu_info;
    AppendCPUTemp(cpu_info);
    cpu_info.WithInteger("idle", idle_diff);
    cpu_info.WithInteger("user", user_diff);
    cpu_info.WithInteger("system", system_diff);
    cpu_info.WithInteger("interrupt", interrupt_diff);
    cpu_info.WithInteger("other", other_diff);
    shadow.WithObject("cpu", cpu_info);
}

void HeartBeatShadow::AppendNetPing(JsonObject &shadow)
{
    long time = ping(pingDst_.c_str());
    if (time < 0)
        return;

    shadow.WithInteger("net_ping", time);
}

void HeartBeatShadow::AppendAppAvailableSpace(JsonObject &shadow)
{
    // use /var/tmp here instead of /var/app/volumes to get
    // a sensible value for tests: this is identical on ABOS
    int64_t free = statvfs_free_kb("/var/tmp");
    if (free < 0)
        return;

    shadow.WithInt64("app_available_space", free);
}

void HeartBeatShadow::AppendMemoryAvailableSpace(JsonObject &shadow)
{
    String out = readFileContents("/proc/meminfo");

    // look for memory available offset
    auto offset = out.find("MemAvailable:");
    if (offset == String::npos)
    {
        AGENT_LOG_WARN("No MemAvailable in /proc/meminfo!");
        return;
    }

    // actually extract size
    out = out.substr(offset + strlen("MemAvailable:"));
    int avail;
    try {
        avail = std::stoi(out.c_str());
    } catch (std::invalid_argument&)
    {
        AGENT_LOG_WARN("/proc/meminfo didn't contain an int after MemAvailable!");
        return;
    }
    shadow.WithInteger("memory_available_space", avail);
}

void HeartBeatShadow::AppendOneDiskIo(JsonObject &shadow, String name)
{
    String stats = readFileContents("/sys/class/block/" + name + "/stat");
    if (stats.empty())
        return;

    /* Fields are described in Documentation/admin-guide/iostats.rst
     * 1 -- # of reads completed (unsigned long)
     * 2 -- # of reads merged, field 6 -- # of writes merged (unsigned long)
     * 3 -- # of sectors read (unsigned long)
     * 4 -- # of milliseconds spent reading (unsigned int)
     * 5 -- # of writes completed (unsigned long)
     * 6 -- # of writes merged (unsigned long)
     * 7 -- # of sectors written (unsigned long)
     * 8 -- # of milliseconds spent writing (unsigned int)
     * 9 -- # of I/Os currently in progress (unsigned int)
     * 10 -- # of milliseconds spent doing I/Os (unsigned int)
     * 11 -- weighted # of milliseconds spent doing I/Os (unsigned int)
     * 12 -- # of discards completed (unsigned long)
     * 13 -- # of discards merged (unsigned long)
     * 14 -- # of sectors discarded (unsigned long)
     * 15 -- # of milliseconds spent discarding (unsigned int)
     * 16 -- # of flush requests completed
     * 17 -- # of milliseconds spent flushing
     * We only pick 3 and 7th (# of sectors read/writes) below.
     */
    std::istringstream iss(stats.c_str());
    unsigned long reads, writes, junk;
    iss >> junk >> junk >> reads;
    iss >> junk >> junk >> junk >> writes;

    if (!iss)
    {
        AGENT_LOG_WARN("/sys/class/block/%s/stat was not as expected: %s",
                       name.c_str(), stats.c_str());
        return;
    }

    // time since last run
    auto now = Monotonic::now();
    float time_diff;
    if (firstRun_)
    {
        time_diff = monotonic_float_since_epoch(now);
    } else
    {
        time_diff = monotonic_float_diff(now, lastRunTs_);
    }

    // adjust with stats of last run if present
    auto prev = diskIoStats_.find(std::string(name.c_str()));
    unsigned long reads_diff = reads, writes_diff = writes;
    if (prev != diskIoStats_.end())
    {
        reads_diff -= prev->second->reads;
        writes_diff -= prev->second->writes;
        prev->second->reads = reads;
        prev->second->writes = writes;
        prev->second->seen = true;
    } else {
        auto stats = new DiskIoStats();
        stats->reads = reads;
        stats->writes = writes;
        stats->seen = true;
        // XXX? Could not get this to work with String or StringView...
        diskIoStats_.insert({std::string(name.c_str()), stats});
    }
    reads_diff *= 1024 / time_diff;
    writes_diff *= 1024 / time_diff;

    JsonObject one_disk_io;
    one_disk_io.WithInteger("read", reads_diff);
    one_disk_io.WithInteger("write", writes_diff);
    shadow.WithObject(name, one_disk_io);
}

void HeartBeatShadow::AppendDiskIo(JsonObject &shadow)
{
    JsonObject disk_io;
    for (const auto &entry : std::filesystem::directory_iterator("/sys/class/block"))
    {
        // mmcblkXbootY and mmcblkXgp* are considered separate but
        // unlikely to be subject to high load: skip them
        String path = entry.path().c_str();
        auto offset = path.find("mmcblk");
        if (offset != String::npos && (
                    path.find("gp", offset+7) != String::npos || 
                    path.find("boot", offset+7) != String::npos))
            continue;
        // partitions don't have a 'device' symlink
        if (!std::filesystem::exists(path + "/device"))
            continue;
        String device = path.substr(path.rfind("/")+1);
        AppendOneDiskIo(disk_io, device);
    }

    // cleanup disks we didn't see (USB?)
    for (auto it = diskIoStats_.cbegin(), next_it = it;
         it != diskIoStats_.end();
         it = next_it)
    {
        next_it++;
        if (!it->second->seen)
        {
            delete(it->second);
            diskIoStats_.erase(it->first);
            continue;
        }
        // reset seen state for next run
        it->second->seen = false;
    }

    shadow.WithObject("disk_io", disk_io);
}

void HeartBeatShadow::AppendEmmcPreEolInfo(JsonObject &shadow)
{
    if (bootDevice_.empty())
        return;

    int pre_eol_info = mmcGetPreEolInfo(bootDevice_.c_str());

    if (pre_eol_info < 0)
        return;

    shadow.WithInteger("emmc_pre_eol_info", pre_eol_info);
}

void HeartBeatShadow::AppendContainerStatus(JsonObject &shadow)
{
    JsonObject tmp_shadow;
    String out, err;
    Vector<String> container_status_list = \
        {"created", "restarting", "running", "removing", "paused", "exited", "dead"};

    executeCommand(
        "podman ps -a --format={{.State}}",
        out,
        err
    );

    for (const String &container_status : container_status_list)
    {
        tmp_shadow.WithInteger(
            container_status,
            CountString(out, container_status));
    }
    shadow.WithObject("container_status", tmp_shadow);
}

void HeartBeatShadow::AppendCellularSiganalQuality(JsonObject &shadow, DBusConnection *dbus)
{
    int signalQuality = modemGetSignalQuality(dbus);

    // 0 means unconnected
    if (signalQuality <= 0)
        return;

    shadow.WithInteger("cellular_signal_quality", signalQuality);
}

void HeartBeatShadow::AppendCellularLocation(JsonObject &shadow, DBusConnection *dbus)
{
    String reply = modemGetLocation(dbus);
    if (reply.empty())
        return;

    // out format is "MCC,MNC,LAC,CID,TAC"
    // replace commas with space to use stringstream parsing...
    std::replace(reply.begin(), reply.end(), ',', ' ');

    std::istringstream iss(reply.c_str());
    unsigned int mcc, mnc, lac, cid, tac;
    iss >> mcc >> mnc;
    iss >> std::hex >> lac >> cid >> tac;

    if (!iss)
    {
        AGENT_LOG_WARN("modem location reply was not 5 words long: %s",
                       reply.c_str());
        return;
    }

    JsonObject tmp_shadow;
    tmp_shadow.WithInteger("MCC", mcc);
    tmp_shadow.WithInteger("MNC", mnc);
    tmp_shadow.WithInteger("LAC", lac);
    tmp_shadow.WithInteger("CID", cid);
    tmp_shadow.WithInteger("TAC", tac);
    shadow.WithObject("lte_location", tmp_shadow);
}

JsonObject DeviceActivityShadow::get(void)
{
    JsonObject  shadow, activity;

    activity.WithString("activity_status", activityStatus_);
    shadow.WithObject("activity", activity);

    return shadow;
}

