/**
 * Copyright (C) 2023-2024 Atmark Techno, Inc. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */

#include <aws/crt/Api.h>
#include <aws/crt/UUID.h>
#include <aws/iot/MqttClient.h>
#include <aws/iotshadow/ErrorResponse.h>
#include <aws/iotshadow/UpdateNamedShadowRequest.h>
#include <aws/iotshadow/UpdateShadowResponse.h>
#include <aws/iotshadow/UpdateNamedShadowSubscriptionRequest.h>

#include "shadow_handler.h"
#include "agent_log.h"

// constructor
ShadowHandler::ShadowHandler(
    String thingName,
    String shadowName)
    : thingName_(thingName)
    , shadowName_(shadowName) {};

ShadowHandler::~ShadowHandler(void)
{
    delete shadowClient_;
}

bool ShadowHandler::Subscribe(std::shared_ptr<Mqtt::MqttConnection> connection)
{
    shadowClient_ = new IotShadowClient(connection);
    std::promise<void> subscribeAcceptedCompletedPromise;
    std::promise<void> subscribeRejectedCompletedPromise;
    bool ret = true;

    AGENT_LOG_DEBUG("Subscribing...");

    auto onUpdateShadowAccepted = [&](UpdateShadowResponse *response, int ioErr)
    {
        if (ioErr != AWS_OP_SUCCESS)
        {
            AGENT_LOG_ERROR("Error on subscription: %s.", ErrorDebugString(ioErr));
            exit(-1);
        }
    };

    auto onUpdateShadowAcceptedSubAck = [&](int ioErr)
    {
        if (ioErr != AWS_OP_SUCCESS)
        {
            AGENT_LOG_ERROR("Error on subscribing to shadow accepted: %s", ErrorDebugString(ioErr));
            exit(-1);
        }
        subscribeAcceptedCompletedPromise.set_value();
    };

    auto onUpdateShadowRejected = [&](ErrorResponse *error, int ioErr)
    {
        if (ioErr != AWS_OP_SUCCESS)
        {
            AGENT_LOG_ERROR("Error on subscription: %s.", ErrorDebugString(ioErr));
            exit(-1);
        }
        AGENT_LOG_WARN("Update of shadow state failed with message %s and code %d.",
                       error->Message->c_str(),
                       *error->Code);
    };

    auto onUpdatedRejectedSubAck = [&](int ioErr)
    {
        if (ioErr != AWS_OP_SUCCESS)
        {
            AGENT_LOG_ERROR("Error subscribing to shadow delta rejected: %s", ErrorDebugString(ioErr));
            exit(-1);
        }
        subscribeRejectedCompletedPromise.set_value();
    };

    UpdateNamedShadowSubscriptionRequest updateNamedShadowSubscriptionRequest;
    updateNamedShadowSubscriptionRequest.ThingName = thingName_;
    updateNamedShadowSubscriptionRequest.ShadowName = shadowName_;

    ret &= shadowClient_->SubscribeToUpdateNamedShadowAccepted(
        updateNamedShadowSubscriptionRequest,
        AWS_MQTT_QOS_AT_LEAST_ONCE,
        onUpdateShadowAccepted,
        onUpdateShadowAcceptedSubAck);

    ret &= shadowClient_->SubscribeToUpdateNamedShadowRejected(
        updateNamedShadowSubscriptionRequest,
        AWS_MQTT_QOS_AT_LEAST_ONCE,
        onUpdateShadowRejected,
        onUpdatedRejectedSubAck);

    subscribeAcceptedCompletedPromise.get_future().wait();
    subscribeRejectedCompletedPromise.get_future().wait();

    return ret;
}

bool ShadowHandler::UpdateShadow(JsonObject shadow, const char* shadowName)
{
    ShadowState state;
    JsonObject desired;
    bool ret = true;

    desired.AsNull();

    AGENT_LOG_DEBUG("Ready to send reported: %s", shadow.View().WriteReadable().c_str());

    state.Desired = desired;
    state.Reported = shadow;

    UpdateNamedShadowRequest updateNamedShadowRequest;
    UUID uuid;
    updateNamedShadowRequest.ClientToken = uuid.ToString();
    updateNamedShadowRequest.ThingName = thingName_;
    updateNamedShadowRequest.State = state;
    updateNamedShadowRequest.ShadowName = (shadowName ? shadowName : shadowName_);

    std::promise<void> publishCompletedPromise;

    auto publishCompleted = [&](int ioErr) {
        if (ioErr != AWS_OP_SUCCESS)
        {
            AGENT_LOG_WARN("Failed to update shadow state: error %s", ErrorDebugString(ioErr));
            return;
        }
        AGENT_LOG_DEBUG("Successfully updated shadow state");
        publishCompletedPromise.set_value();
    };

    ret = shadowClient_->PublishUpdateNamedShadow(
        updateNamedShadowRequest,
        AWS_MQTT_QOS_AT_LEAST_ONCE,
        std::move(publishCompleted));

    publishCompletedPromise.get_future().wait();
    return ret;
}
