//
// power_notification_handler.cpp
//
#include "power_notification_handler.h"

#include <algorithm>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <poll.h>
#include <signal.h>
#include <time.h>
#include <unistd.h>

#include <thread>

#include "agent_log.h"
#include "agent_utils.h"


#define SERVICE_NAME	"atmark-power-svc"
#define REQ_POWER_STATE_NOTIFICATION	"request_power_state_notificaiton"
#define NFY_ENTER_SLEEP	"enter_sleep"
#define NFY_LEAVE_SLEEP	"leave_sleep"

#define ADDR_UN_LEN(addr_un)	(sizeof((addr_un).sun_family) + strlen((addr_un).sun_path + 1) + 2)

#define SLEEP_MILLIS	100


//
// internal class
//
class ExitTimer {
public:
    ExitTimer() : thread_(), quitRequested_(false), downCounter_(0) {}
    ~ExitTimer() { this->Stop(); }

    void	Start(unsigned int timeoutSec, const char* caller);
    void	Stop();
    bool	IsRunning() const { return thread_.joinable(); }

private:
    static void	Run(ExitTimer* obj, const char* caller);

    std::thread	thread_;
    bool	quitRequested_;
    unsigned int	downCounter_;
};

void
ExitTimer::Start(unsigned int timeoutSec, const char* caller)
{
    if (this->IsRunning()) {
        AGENT_LOG_ERROR("ExitTimer is already running.");
        return;
    }
    quitRequested_ = false;
    downCounter_ = timeoutSec * (1000 / SLEEP_MILLIS);
    thread_ = std::thread(Run, this, caller);
}

void
ExitTimer::Stop()
{
    if (this->IsRunning()) {
        quitRequested_ = true;
        thread_.join();
    }
}

void
ExitTimer::Run(ExitTimer* obj, const char* caller)
{
    while (! obj->quitRequested_) {
        std::this_thread::sleep_for(std::chrono::milliseconds(SLEEP_MILLIS));
        if (obj->downCounter_ > 0) {
            obj->downCounter_--;
        } else {
            AGENT_LOG_ERROR("ExitTimer expired; caller= %s", caller);
            std::exit(-1);
        }
    }
}


//
// internal functions
//
static void
init_pollfd(struct pollfd* ioPollFd, int targetFd)
{
    ioPollFd->fd = targetFd;
    ioPollFd->events = POLLIN;
    ioPollFd->revents = 0;
}

static inline uint64_t
timespec2millis(const struct timespec& ts)
{
    return (ts.tv_sec * 1000 + ts.tv_nsec / 1000000);
}

static uint64_t
GetCurrentTimeInMillis()
{
    struct timespec  ts;

    (void)clock_gettime(CLOCK_BOOTTIME, &ts);

    return timespec2millis(ts);
}

static void
MakeSockAddr(struct sockaddr_un* addrBuf, const char* name)
{
    bzero(addrBuf, sizeof(*addrBuf));
    addrBuf->sun_family = AF_LOCAL;
    strcpy((char*)addrBuf->sun_path + 1, name);
}

static int
MakeNamedSocket(const char* name)
{
    int  sockfd = socket(AF_LOCAL, SOCK_DGRAM|SOCK_CLOEXEC, 0);
    struct sockaddr_un  sockAddr;

    if (sockfd < 0) {
        return -1;
    }
    bzero(&sockAddr, sizeof(sockAddr));
    sockAddr.sun_family = AF_LOCAL;
    strcpy((char*)sockAddr.sun_path + 1, name);
    // NOTE: name (: client_id) is shorter than
    //       sun_path (: UNIX_PATH_MAX) i.e. 64 < 108
    if (bind(sockfd, (struct sockaddr*)&sockAddr, ADDR_UN_LEN(sockAddr)) < 0) {
        (void)close(sockfd);
        return -1;
    }

    return sockfd;
}

static bool
RequestPowerStateNotification(int sockfd)
{
    bool  isOK = false;
    struct sockaddr_un  serverAddr;
    const char*  msg = REQ_POWER_STATE_NOTIFICATION;

    MakeSockAddr(&serverAddr, SERVICE_NAME);
    if (sendto(sockfd, msg, strlen(msg) + 1, 0,
            (struct sockaddr*)&serverAddr, ADDR_UN_LEN(serverAddr)) < 0) {
        AGENT_LOG_ERROR("failed to sendto(), errno= %d, serverr= '%s'.",
            errno, serverAddr.sun_path + 1);
    } else {
        char	msgBuf[64] = { 0 };
        struct sockaddr_un	rmtAddr;
        socklen_t	len = sizeof(rmtAddr);
        struct pollfd  pollfdSet[1];

        init_pollfd(&pollfdSet[0], sockfd);
        if (poll(pollfdSet, 1, 10) <= 0) {
            AGENT_LOG_ERROR("timed out for waiting the reply from power-utils");
            return false;
        }
        if (recvfrom(sockfd, msgBuf, sizeof(msgBuf) - 1, 0, (struct sockaddr*)&rmtAddr, &len) < 0) {
            AGENT_LOG_ERROR("failed to recvfrom().");
        } else if (0 != strcmp(msgBuf, "OK")) {
            AGENT_LOG_ERROR("request rejected, reply: '%s'", msgBuf);
        } else {
            isOK = true;
        }
    }

    return isOK;
}

static int
GetPowerUtilsPid()
{
    String  pidStr = readFileContents("/var/run/power-utils.pid");

    if (pidStr.empty()) {
        AGENT_LOG_ERROR("power-utils daemon seems not running!");
        return -1;
    }

    return stoi_safe(pidStr);
}


//
// PowerNotificationHandler static data member
//
PowerNotificationHandler*  PowerNotificationHandler::singleton = NULL;
std::mutex  PowerNotificationHandler::mtx;


//
// PowerNotificationHandler public methods
//
PowerNotificationHandler::PowerNotificationHandler()
    : monitorSock_(-1), hasPendingSleepConfirmation_(false), rmtAddrLen_(0), inhibitSleepCount_(0),
    powerUtilsPid_(-1), powerUtilsAlive_(false)
{
    std::lock_guard  lock(mtx);
    char  client_id[64];
    int  sockfd;

    if (NULL != singleton) {
        AGENT_LOG_ERROR("already create the instance of PowerNotificationHandler");
        return;
    }

    sprintf(client_id, "atmark-power-listener_%d", getpid());
    sockfd = MakeNamedSocket(client_id);
    if (sockfd < 0) {
        AGENT_LOG_ERROR("failed to MakeNamedSocket().");
        return;
    }

    if (!RequestPowerStateNotification(sockfd)) {
        AGENT_LOG_ERROR("failed to RequestPowerStateNotification().");
    } else {
        powerUtilsPid_ = GetPowerUtilsPid();
        powerUtilsAlive_ = true;
    }
    monitorSock_ = sockfd;
    singleton = this;
}

PowerNotificationHandler::~PowerNotificationHandler()
{
    if (this->IsValid()) {
        (void)close(monitorSock_);
        singleton = NULL;
    }
}

bool
PowerNotificationHandler::WaitSleepNotification(const struct timespec& endTime)
{
    return this->DoWaitSleepNotification(endTime, false);
}

bool
PowerNotificationHandler::CheckSleepNotification()
{
    struct timespec	time_val;

    (void)clock_gettime(CLOCK_BOOTTIME, &time_val);
    return this->DoWaitSleepNotification(time_val, false);  // endTime == now
}

void
PowerNotificationHandler::ConfirmSleepAndWaitResume()
{
    if (! hasPendingSleepConfirmation_) {
        return;  // assertion
    }

    hasPendingSleepConfirmation_ = false;
    if (sendto(monitorSock_, "done", strlen("done") + 1, 0,
            (sockaddr*)&rmtAddress_, rmtAddrLen_) < 0) {
        AGENT_LOG_ERROR("failed to sendto().");
    } else {
        char	msgBuf[64] = { 0 };
        ExitTimer   exitTimer;
        exitTimer.Start(600, __func__);  // it will automatically call Stop() by the destrcutor
        int n = recvfrom(monitorSock_, msgBuf, sizeof(msgBuf) - 1, 0,
            (sockaddr*)&rmtAddress_, &rmtAddrLen_);

        if (n > 0) {
            if (strcmp(msgBuf, NFY_LEAVE_SLEEP) != 0) {
                AGENT_LOG_ERROR("unexpected notification while waiting the resume: '%s'", msgBuf);
            }
        }
    }
}

void
PowerNotificationHandler::WaitWithAutoSleepConfirm(uint32_t secs)
{
    struct timespec	time_val;

    (void)clock_gettime(CLOCK_BOOTTIME, &time_val);
    time_val.tv_sec += secs;
    (void)this->DoWaitSleepNotification(time_val, true);
}

bool
PowerNotificationHandler::InhibitSleep()
{
    std::lock_guard  lock(mtx);

    if (inhibitSleepCount_ == 0 && powerUtilsAlive_) {
        ExitTimer   exitTimer;
        exitTimer.Start(10, __func__);  // it will automatically call Stop() by the destrcutor
        if (0 != system("/usr/bin/power-utils inhibit_sleep")) {
            if (kill(powerUtilsPid_, 0) < 0 && errno == ESRCH) {
                powerUtilsAlive_ = false;  // mark dead and don't return false
            } else {
                return false;
            }
        }
    }
    ++inhibitSleepCount_;
    return true;
}

void
PowerNotificationHandler::AllowSleep()
{
    std::lock_guard  lock(mtx);

    if (inhibitSleepCount_ == 1 && powerUtilsAlive_) {
        ExitTimer   exitTimer;
        exitTimer.Start(10, __func__);  // it will automatically call Stop() by the destrcutor
        (void)system("/usr/bin/power-utils allow_sleep");
    } else if (inhibitSleepCount_ <= 0) {
        AGENT_LOG_ERROR("invalid status; inhibitSleepCount_= %d", inhibitSleepCount_);
        return;
    }
    --inhibitSleepCount_;
}

//
// PowerNotificationHandler static method
//
PowerNotificationHandler*
PowerNotificationHandler::Singleton()
{
    return singleton;
}

//
// PowerNotificationHandler private method
//
bool
PowerNotificationHandler::DoWaitSleepNotification(const struct timespec& endTime, bool doConfirm)
{
    uint64_t  endTimeMillis = timespec2millis(endTime);
    struct pollfd  pollfdSet[1];
    char  msgBuf[64];
    int  timeout = (int)(endTimeMillis - std::min(endTimeMillis, GetCurrentTimeInMillis()));

    init_pollfd(&pollfdSet[0], monitorSock_);
    do {
        bzero(&rmtAddress_, sizeof(rmtAddress_));
        rmtAddrLen_ = sizeof(rmtAddress_);
        if (poll(pollfdSet, 1, timeout) > 0) {
            int n = recvfrom(monitorSock_, msgBuf, sizeof(msgBuf) - 1, 0,
                (sockaddr*)&rmtAddress_, &rmtAddrLen_);

            if (n > 0) {
                msgBuf[n] = '\0';
                if (strcmp(msgBuf, NFY_ENTER_SLEEP) == 0) {
                    if (doConfirm) {
                        if (sendto(monitorSock_, "done", strlen("done") + 1, 0,
                                (sockaddr*)&rmtAddress_, rmtAddrLen_) < 0) {
                            AGENT_LOG_ERROR("failed to sendto().");
                        }
                    } else {
                        hasPendingSleepConfirmation_ = true;
                        return true;
                    }
                } else if (! doConfirm) {
                    AGENT_LOG_ERROR("unexpected notification: '%s'", msgBuf);
                }
            } else {
                AGENT_LOG_ERROR("failed to recvfrom().");
            }
            pollfdSet[0].revents = 0;
        }
        timeout = (int)(endTimeMillis - std::min(endTimeMillis, GetCurrentTimeInMillis()));
    } while (timeout > 0);

    return false;
}

//
// End of File
//
