/**
 * Copyright (C) 2023-2024 Atmark Techno, Inc. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */

#include "mqtt_handler.h"
#include "agent_log.h"

// constructor
MqttHandler::MqttHandler(
    String thingName,
    std::shared_ptr<Mqtt::MqttConnection> connection)
    : thingName_(thingName)
    , connection_(connection) {};

int MqttHandler::Connect(void)
{
    std::promise<int> connectionCompletedPromise;

    auto onConnectionCompleted = [&](Mqtt::MqttConnection &, int errorCode, Mqtt::ReturnCode returnCode, bool) {
        AGENT_LOG_DEBUG("onConnectionCompleted: ");
        if (errorCode)
        {
            AGENT_LOG_ERROR("Connection failed with error %s", ErrorDebugString(errorCode));
        }
        else
        {
            AGENT_LOG_DEBUG("Connection completed with return code %d", returnCode);
        }
        connectionCompletedPromise.set_value(errorCode);
    };

    auto onDisconnect = [&](Mqtt::MqttConnection &) {
        AGENT_LOG_DEBUG("onDisconnect: Disconnect completed");
        connectionClosedPromise_.set_value();
    };

    auto onInterrupted = [&](Mqtt::MqttConnection &, int error) {
        if (error != AWS_ERROR_MQTT_TIMEOUT) {
            AGENT_LOG_DEBUG("Connection interrupted, error= %d", error);
            AGENT_LOG_ERROR("exit when connnection interrupted");
            exit(-1);
        }
    };

    connection_->OnConnectionCompleted = std::move(onConnectionCompleted);
    connection_->OnDisconnect = std::move(onDisconnect);
    connection_->OnConnectionInterrupted = std::move(onInterrupted);

    AGENT_LOG_DEBUG("Connecting...");
    if (!connection_->Connect(thingName_.c_str(), false, 120))
    {
        AGENT_LOG_ERROR("MQTT Connection failed with error %s", ErrorDebugString(connection_->LastError()));
        return -1;
    }
    return connectionCompletedPromise.get_future().get();
}

void MqttHandler::Disconnect(void)
{
    connection_->Disconnect();
    connectionClosedPromise_.get_future().wait();
}
