/**
 * Copyright (C) 2023-2024 Atmark Techno, Inc. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */

#include "command_line_utils.h"
#include "mqtt_handler.h"
#include "shadow_handler.h"
#include "shadow_factory.h"
#include "job_handler.h"
#include "agent_utils.h"
#include "agent_config.h"
#include "agent_log.h"
#include "power_notification_handler.h"

#define SHADOW_NAME "heart_beat"
#define HEARTBEAT_INTERVAL 300 // seconds
#define DEFAULT_ENDPOINT "a2cxtlwgru3aml-ats.iot.us-east-1.amazonaws.com"
#define DEFAULT_CREDENTIAL_PROVIDER_ENDPOINT "c1vhtm72uows73.credentials.iot.us-east-1.amazonaws.com"
#define DEFAULT_CREDENTIAL_PROVIDER_ENDPOINT_SUFFIX "/role-aliases/atmark-atwin-iot-device-iotsjob-iot-role-alias-01/credentials"
#define DEFAULT_IOT_JOB_ENDPOINT "lhus6wg16bdqj.jobs.iot.us-east-1.amazonaws.com"
#define DEFAULT_RECONNECT_INTERVAL 10 // seconds
#define MAX_RECONNECT_INTERVAL 60 * 60 * 24 // 1day (seconds)
#define LOGGING_HEARTBEAT_PER_DAY 4
#define LOGGING_HEARTBEAT_INTERVAL_COUNT static_cast<int>((24 * 60 * 60 / HEARTBEAT_INTERVAL) / LOGGING_HEARTBEAT_PER_DAY)
#define STATUS_STARTING   "starting"
#define STATUS_TRY_CONNECTING   "try_connecting"
#define STATUS_CONNECTED    "connected"
#define STATUS_SLEEPING "sleeping"
#define STATUS_RESUMING "resuming"
#define STATUS_FILE_PATH    "/var/run/armadillo-twin-agent.status"

using namespace Aws::Crt;


static void WriteCurrentStatus(const char* status)
{
    const char* tmpPath = STATUS_FILE_PATH ".tmp";
    FILE* fp = fopen(tmpPath, "w");
    if (fp == NULL)
    {
        AGENT_LOG_ERROR("failed to open %s.", tmpPath);
        return;
    }
    fprintf(fp, "%s\n", status);
    fclose(fp);
    (void)rename(tmpPath, STATUS_FILE_PATH);
}


static void NotifySleepToTwin(ShadowHandler* shadowHandler)
{
    DeviceActivityShadow    shadow("sleep");
    shadowHandler->UpdateShadow(shadow.get(), "activity");
}


static void TryInhibitSleep(PowerNotificationHandler* powerNotificationHandler, ShadowHandler* shadowHandler)
{
    if (!powerNotificationHandler->InhibitSleep())
    {
        if (powerNotificationHandler->CheckSleepNotification())
        {
            if (NULL != shadowHandler)
            {
                NotifySleepToTwin(shadowHandler);
            }
            powerNotificationHandler->ConfirmSleepAndWaitResume();
            if (!powerNotificationHandler->InhibitSleep())
            {
                goto inhibit_sleep_failed;
            }
        }
        else
        {
inhibit_sleep_failed:
            AGENT_LOG_ERROR("failed to PowerNotificationHandler::InhibitSleep()");
        }
    }
}

int main(int argc, char *argv[])
{
    ApiHandle apiHandle;
    ::WriteCurrentStatus(STATUS_STARTING);

    // command line arguments handling
    Utils::CommandLineUtils cmdUtils = Utils::CommandLineUtils();
    cmdUtils.InitCommands();
    const char **const_argv = (const char **)argv;
    cmdUtils.SendArguments(const_argv, const_argv + argc);
    cmdUtils.StartLoggingBasedOnCommand(&apiHandle);

    if (!checkAndCreatePidFile())
        exit(-1);

    AgentConfig agentConfig;
    AGENT_LOG_SET_CONSOLE_LOGLEVEL(agentConfig.get("output-console-loglevel"));
    AGENT_LOG_SET_LOGFILE_LOGLEVEL(agentConfig.get("output-logfile-loglevel"));

    String endpoint = agentConfig.get("endpoint");
    if (endpoint.empty())
    {
        AGENT_LOG_DEBUG("failed to get endpoint address from config file, use default endpoint");
        endpoint = DEFAULT_ENDPOINT;
    }
    cmdUtils.SetEndpoint(endpoint);
    bool isCommandExecAllowed = false;
    if (agentConfig.get("allow-exec-command") == "yes")
    {
        isCommandExecAllowed = true;
    }
    AGENT_LOG_DEBUG("exec-command feature status: %s",
                    isCommandExecAllowed ? "Enabled" : "Disabled");

    // environment variables for SWUpdate
    String credentialProviderEndpoint = agentConfig.get("credential-provider-endpoint");
    if (credentialProviderEndpoint.empty())
    {
        credentialProviderEndpoint = String(DEFAULT_CREDENTIAL_PROVIDER_ENDPOINT);
    }
    String credentialProviderURL = credentialProviderEndpoint + DEFAULT_CREDENTIAL_PROVIDER_ENDPOINT_SUFFIX;
    if (setenv("CREDENTIAL_PROVIDER_URL", credentialProviderURL.c_str(), 1) != 0)
    {
        AGENT_LOG_WARN("failed to set CREDENTIAL_PROVIDER_URL environment variable");
    }
    String iotJobEndpoint = agentConfig.get("iot-job-endpoint");
    if (iotJobEndpoint.empty())
    {
        iotJobEndpoint = String(DEFAULT_IOT_JOB_ENDPOINT);
    }
    if (setenv("IOT_JOB_ENDPOINT", iotJobEndpoint.c_str(), 1) != 0)
    {
        AGENT_LOG_WARN("failed to set IOT_JOB_ENDPOINT environment variable");
    }

    // create instances
    String thingName = cmdUtils.GetCommandRequired("thing_name");
    MqttHandler* mqttHandler = nullptr;
    ShadowHandler* shadowHandler = new ShadowHandler(thingName, SHADOW_NAME);
    ShadowFactory* shadow_factory = new ShadowFactory(endpoint);
    JobHandler* jobHandler = nullptr;
    PowerNotificationHandler    powerNotificationHandler;

    if (! powerNotificationHandler.IsValid()) {
        AGENT_LOG_ERROR("failed to create instance of PowerNotificationHandler");
        exit(-1);
    }

    // MQTT setup
    std::shared_ptr<Mqtt::MqttConnection> connection = cmdUtils.BuildMQTTConnection();

    mqttHandler = new MqttHandler(thingName, connection);
    if (mqttHandler == nullptr)
    {
        AGENT_LOG_ERROR("failed to create instance of MqttHandler");
        exit(-1);
    }

    int reconnectInterval = DEFAULT_RECONNECT_INTERVAL;

    ::WriteCurrentStatus(STATUS_TRY_CONNECTING);
    while (true)
    {
        int errCode = mqttHandler->Connect();
        if (errCode == AWS_ERROR_SUCCESS)
        {
            break;
        }
        else if (errCode == AWS_ERROR_PKCS11_CKR_KEY_HANDLE_INVALID
          || errCode == AWS_ERROR_PKCS11_CKR_SESSION_HANDLE_INVALID)
        {
            AGENT_LOG_ERROR("failed to connect about PCKS11_CKR_[KEY|SESSION]");
            exit(-1);
        /* These error occurrence means it may be the session with SE050
         * have interrupted by other program's SE050 access. Typical case is
         * that other program invokes se05x-getkey command.
         * If these error has occurred, it must doing the connection sequence
         * from the beggining that is to say recreate Mqtt::MqttConenction object
         * and MqttHandler object then make connection again. Of cource it must
         * delete old  objects and recreate other stuffs.
         * But for simplicity, do self exit and rely automatic restart by OpenRC here.
         */
        }

        AGENT_LOG_WARN("failed to connect to AWS. reconnect after %d seconds", reconnectInterval);
        powerNotificationHandler.WaitWithAutoSleepConfirm(reconnectInterval);

        /* If an unprovisioned device attempts to connect to Twin and fails,
         * extend the reconnection interval (up to 1 day)
         *
         * from https://docs.aws.amazon.com/iot/latest/developerguide/device-advisor-troubleshooting.html
         * The AWS_ERROR_MQTT_UNEXPECTED_HANGUP error can occur under
         * a variety of conditions, but is most often described as
         * being due to AWS thing role.
         */
        if (errCode == AWS_ERROR_MQTT_UNEXPECTED_HANGUP &&
            reconnectInterval <= MAX_RECONNECT_INTERVAL)
        {
            reconnectInterval = std::min(reconnectInterval * 2, MAX_RECONNECT_INTERVAL);
        }
    }
    ::WriteCurrentStatus(STATUS_CONNECTED);

    // inhibit sleep until sending the heartbeat
    TryInhibitSleep(&powerNotificationHandler, NULL);

    // subscribe topics of IoT shadow
    if (!shadowHandler->Subscribe(connection))
    {
        AGENT_LOG_ERROR("failed to subscribe to IoT shadow");
        exit(-1);
    }

    // subscribe topics of IoT Jobs
    jobHandler = new JobHandler(thingName, isCommandExecAllowed, connection);
    if (jobHandler == nullptr)
    {
        AGENT_LOG_ERROR("failed to create instance of JobHandler");
        exit(-1);
    }
    if (!jobHandler->Subscribe())
    {
        AGENT_LOG_ERROR("failed to IoT jobs subscribe");
        exit(-1);
    }
    jobHandler->InitialJobExec();

    // get /etc/sw-versions before starting job thread
    String swVersions = readFileContents("/etc/sw-versions");

    // start job thread
    (void)jobHandler->StartMainLoop();

    // upload device info
    Shadow *device_info_shadow = shadow_factory->CreateShadow(ShadowPurpose::DeviceInfo, isCommandExecAllowed);
    if (!shadowHandler->UpdateShadow(device_info_shadow->get()))
    {
        AGENT_LOG_ERROR("failed to update shadow");
        exit(-1);
    }

    // main loop
    int times_sent_heartbeat = 0;
    Shadow *heart_beat_shadow = shadow_factory->CreateShadow(ShadowPurpose::HeartBeat);
    while (true)
    {
        // Resend device_info when /etc/sw-versions is updated without reboot
        // by SWUpdate with POST_ACTION=container
        String newSwVersions = readFileContents("/etc/sw-versions");
        if (newSwVersions != swVersions)
        {
            shadowHandler->UpdateShadow(device_info_shadow->get());
            swVersions = newSwVersions;
        }

        // upload heartbeat
        JsonObject heart_beat = heart_beat_shadow->get();
        struct timespec	time_val;
        (void)clock_gettime(CLOCK_BOOTTIME, &time_val);
        if (!shadowHandler->UpdateShadow(heart_beat))
        {
            AGENT_LOG_ERROR("failed to update shadow");
            exit(-1);
        }
        ::WriteCurrentStatus(STATUS_CONNECTED);
        if (times_sent_heartbeat == 0)
            AGENT_LOG_INFO("%s", heart_beat.View().WriteCompact().c_str());
        times_sent_heartbeat = (times_sent_heartbeat + 1) % LOGGING_HEARTBEAT_INTERVAL_COUNT;
        time_val.tv_sec += HEARTBEAT_INTERVAL;
        powerNotificationHandler.AllowSleep();
        while (powerNotificationHandler.WaitSleepNotification(time_val)) {
            ::WriteCurrentStatus(STATUS_SLEEPING);
            NotifySleepToTwin(shadowHandler);
            powerNotificationHandler.ConfirmSleepAndWaitResume();
            ::WriteCurrentStatus(STATUS_RESUMING);
        }
        TryInhibitSleep(&powerNotificationHandler, shadowHandler);
    }
    delete device_info_shadow;
    delete heart_beat_shadow;

    // disconnect
    AGENT_LOG_DEBUG("Disconnecting...");
    mqttHandler->Disconnect();
    AGENT_LOG_DEBUG("Disconnected!");

    jobHandler->StopMainLoop();
    delete shadow_factory;
    delete shadowHandler;
    delete mqttHandler;
    delete jobHandler;
    return 0;
}
