#pragma once
/**
 * Copyright (C) 2023-2024 Atmark Techno, Inc. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */

#include <aws/crt/Api.h>
#include <aws/crt/UUID.h>
#include <aws/iotjobs/JobExecutionsChangedEvent.h>
#include <aws/iotjobs/JobExecutionsChangedSubscriptionRequest.h>
#include <aws/iotjobs/UpdateJobExecutionRequest.h>
#include <aws/iotjobs/UpdateJobExecutionResponse.h>
#include <aws/iotjobs/UpdateJobExecutionSubscriptionRequest.h>
#include <aws/iotjobs/GetPendingJobExecutionsResponse.h>
#include <aws/iotjobs/GetPendingJobExecutionsSubscriptionRequest.h>
#include <aws/iotjobs/GetPendingJobExecutionsRequest.h>
#include <aws/iotjobs/DescribeJobExecutionRequest.h>
#include <aws/iotjobs/DescribeJobExecutionResponse.h>
#include <aws/iotjobs/DescribeJobExecutionSubscriptionRequest.h>
#include <aws/iotjobs/IotJobsClient.h>
#include <aws/iotjobs/RejectedError.h>

#include <mutex>
#include <sys/stat.h>
#include <filesystem>

#include "execute_command.h"
#include "agent_utils.h"

using namespace Aws::Crt;
using namespace Aws::Iotjobs;
namespace fs = std::filesystem;

class JobHandler
{
public:
    JobHandler(
        String thingName,
        bool isCommandExecAllowed,
        std::shared_ptr<Mqtt::MqttConnection> connection);
    ~JobHandler(void);
    void MainLoop(void);
    bool StartMainLoop(void);
    void StopMainLoop(void);
    bool Subscribe(void);
    void InitialJobExec(void);

private:
    std::mutex mtx_;
    std::mutex mtxUpdateJobState_;
    std::promise<void> jobExecutionsChangedPromise_;
    std::promise<Vector<JobExecutionSummary>> getInProgressJobsPromise_;
    std::promise<Vector<JobExecutionSummary>> getQueuedJobsPromise_;
    std::promise<JobExecutionData> jobExecutionDataPromise_;
    std::promise<std::optional<RejectedError>> finishUpdateJobStatePromise_;
    const String agentStartedFilePath_ = "/run/armadillo_twin_agent_started";
    const String swuInstallDoneFilePath_ = "/var/log/swupdate/armadillo_twin_install_done";
    const String swuWaitingRebootFilePath_ = "/run/swupdate_waiting";
    String jobIdExecutingSwupdate_ = "";
    String jobIdExecutingCommand_ = "";
    bool isInitialJobGet_ = true;
    String thingName_;
    bool isCommandExecAllowed_ = false;
    std::shared_ptr<Mqtt::MqttConnection> connection_;
    IotJobsClient* jobClient_ = nullptr;
    bool isQuitRequested_ = false;
    std::thread  mainLoopThread_;
    Vector<JobExecutionSummary> initialQueuedJobs_;

    void SetJobIdExecutingCommand(String jobId);
    String GetJobIdExecutingCommand(void);
    void SetJobIdExecutingSwupdate(String jobId);
    String GetJobIdExecutingSwupdate(void);
    bool SubscribeJobExecutionsChanged(void);
    bool SubscribeUpdateJobExecution(void);
    bool SubscribeGetPendingJobExecutions(void);
    bool SubscribeDescribeJobExecution(void);
    void SubscribeCanceledJob(String jobId);
    void UnsubscribeCanceledJob(String jobId);
    bool GetPendingJobExecutions(
        Vector<JobExecutionSummary>& queued,
        Vector<JobExecutionSummary>& inProgress);
    bool GetDescribeJobExecution(
        Vector<JobExecutionSummary> jobSummaries,
        Vector<JobExecutionData>& jobExecutions);
    void RunJobSteps(String jobId, JsonObject jobStepObj);
    void RunJobsFromSummary(Vector<JobExecutionSummary> jobSummaries);
    std::optional<RejectedError> UpdateJobState(
        String jobId,
        String jobStatus,
        Map<String, String> statusDetails);
    void RunJob(
        String jobId,
        JsonView jobSteps);
    void RunCommand(String jobId, JsonView actionInput);
    void ExecSWUpdate(
        String jobId,
        JsonView actionInput);
    bool CheckInstallDoneFile(String jobId);
};
