/**
 * Copyright (C) 2023-2024 Atmark Techno, Inc. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
#pragma once
#include <aws/crt/Types.h>
#include <mutex>

#define AGENT_LOG_FILEPATH "/var/log/armadillo-twin-agent/agent_log"
#define MAX_LOG_SIZE (1 * 1024 * 1024)  // bytes
#define MAX_LOG_GENERATION 1
#define DEFAULT_CONSOLE_LOGLEVEL AgentLogLevel::DEBUG
#define DEFAULT_LOGFILE_LOGLEVEL AgentLogLevel::INFO
#define AGENT_LOG_DEBUG(...) AgentLogger::instance().log(AgentLogLevel::DEBUG, __VA_ARGS__)
#define AGENT_LOG_INFO(...) AgentLogger::instance().log(AgentLogLevel::INFO, __VA_ARGS__)
#define AGENT_LOG_WARN(...) AgentLogger::instance().log(AgentLogLevel::WARN, __VA_ARGS__)
#define AGENT_LOG_ERROR(...) AgentLogger::instance().log(AgentLogLevel::ERROR, __VA_ARGS__)
#define AGENT_LOG_SET_CONSOLE_LOGLEVEL(level) AgentLogger::instance().setConsoleLogLevel(level)
#define AGENT_LOG_SET_LOGFILE_LOGLEVEL(level) AgentLogger::instance().setLogFileLogLevel(level)

using namespace Aws::Crt;

enum AgentLogLevel
{
    DEBUG,
    INFO,
    WARN,
    ERROR,
    NONE
};

static const std::map<String, AgentLogLevel> levelStringMap = {
    {"DEBUG", AgentLogLevel::DEBUG},
    {"INFO", AgentLogLevel::INFO},
    {"WARN", AgentLogLevel::WARN},
    {"ERROR", AgentLogLevel::ERROR},
    {"NONE", AgentLogLevel::NONE}
};

class AgentLogger
{
public:
    static AgentLogger& instance(void);
    void setConsoleLogLevel(String level);
    void setLogFileLogLevel(String level);
    void log(AgentLogLevel level, const char* format, ...);

private:
    AgentLogLevel consoleLogLevel_ = DEFAULT_CONSOLE_LOGLEVEL;
    AgentLogLevel logFileLogLevel_ = DEFAULT_LOGFILE_LOGLEVEL;
    std::mutex mtx_;
    String logLevelToString(AgentLogLevel level);
    bool stringToLogLevel(const String& level, AgentLogLevel& agentLogLevel);
    String getCurrentTime(void);
    void logRotate(const String& logFilePath, const size_t logFileMaxSize);
};
