#!/bin/sh
# SPDX-License-Identifier: MIT

. "$(dirname "$0")/common"

ntp_info_in_sync() {
	# we regroup all ntp tests because we want to get infos at various points
	# to verify get ouput
	# assume the test runs from a clean, synchronized state
	# TODO: try to double-check through ssh if this fails?
	req_bearer "/api/time/ntp_info" \
		|| error "Could not get ntp info"
	case "$(jq -r ".ntp_server_ip" < req)" in
	[0-9a-f]*) ;; # probably an IP
	*) error "ntp_server_ip didn't look like an IP";;
	esac
	case "$(jq -r ".ntp_server_offset" < req)" in
	-0.0*|0.0*) ;; # in sync
	*) error "not synced within 100ms to server?";;
	esac
	local now_armadillo now_local diff
	now_local=$(date +%s) || error "Could not get local timestamp"
	now_armadillo=$(jq -re ".time_now" < req) \
		|| error "Could not get armadillo time"
	diff=$((now_local - now_armadillo)) \
		|| error "Either local or armadillo time was not an int?"
	[ "${diff#-}" -lt 10 ] || error "More than 10s difference between local and armadillo"
}

force_sync_test() {
	# run this before ntp tests so time will be fixed with ntp tests
	# use one of three forms at random so we eventually get coverage
	local post offset=0
	for post in urlencode json-int json-string; do
		offset=$((offset+3600))
		case $post in
		urlencode)
			req_bearer "/api/time/set" -X POST -d timestamp=$(($(date +%s) - offset))
			;;
		json-int)
			req_bearer "/api/time/set" -X POST -H "Content-type: application/json" \
				--data-raw '{"timestamp": '"$(($(date +%s) - offset))"'}'
			;;
		*)
			req_bearer "/api/time/set" -X POST -H "Content-type: application/json" \
				--data-raw '{"timestamp": "'"$(($(date +%s) - offset))"'"}'
			;;
		esac || error "Could not set time $((offset/3600))h back ($post)"

		req_bearer "/api/time/ntp_info" \
			|| error "Could not get ntp info after changing date"
		local now_armadillo now_local diff
		now_local=$(date +%s) || error "Could not get local timestamp"
		now_armadillo=$(jq -re ".time_now" < req) \
			|| error "Could not get armadillo time ($((offset/3600))h offset)"
		diff=$((now_local - now_armadillo - offset)) \
			|| error "Either local or armadillo time was not an int?"
		[ "${diff#-}" -lt 10 ] \
			|| error "More than 10s difference between local and armadillo ($post): $diff"
	done
}

ntp_tests() {
	# setting off values ('off' or empty string)
	# note we keep initstepslew off to save time on restarts
	req_bearer "/api/time/ntp_config" -X POST -d servers=off -d initstepslew= \
		|| error "Could not reset ntp config to off"
	check_req_jq no_server '.servers[]' ""
	check_req_jq no_slew .initstepslew null

	# get ntp info again immediately after restart: should not be sync'd
	req_bearer "/api/time/ntp_info" \
		|| error "Could not get ntp info"
	check_req_jq no_ip .ntp_server_ip null
	check_req_jq no_offset .ntp_server_offset null

	# custom values
	req_bearer "/api/time/ntp_config" -X POST -d servers="pool pool.ntp.org" \
			-d servers="server 192.168.1.1" \
		|| error "Could not reset ntp config to off"
	check_req_jq two_servers '.servers[]' "pool pool.ntp.org
server 192.168.1.1"
	check_req_jq no_slew .initstepslew null

	req_bearer "/api/time/ntp_config" -X POST -d servers="pool pool.ntp.org" \
		|| error "Could not reset ntp config to off"
	check_req_jq servers_pool '.servers[]' "pool pool.ntp.org"
	check_req_jq no_slew .initstepslew null

	req_bearer "/api/time/ntp_config" -X POST -d initstepslew="1" \
		|| error "Could not reset ntp config to off"
	check_req_jq servers_pool '.servers[]' "pool pool.ntp.org"
	check_req_jq slew_1 .initstepslew 1

	# default values
	req_bearer "/api/time/ntp_config" -X POST -d servers=default -d initstepslew=default \
		|| error "Could not reset ntp config to default"
	check_req_jq servers_default '.servers[]' "pool pool.ntp.org iburst"
	check_req_jq slew_default .initstepslew '10 pool.ntp.org'
}

timezone_test() {
	req_bearer "/api/time/timezone" -X POST -d timezone=Europe/Paris \
		|| error "Could not set timezone"

	req_bearer "/api/time/timezone" \
		|| error "Could not get timezone"
	check_req_jq timezone .timezone 'Europe/Paris'

	req_bearer "/api/time/timezone" -X POST -d timezone=Asia/Tokyo \
		|| error "Could not restore timezone"
}

run_tests ntp_info_in_sync force_sync_test ntp_tests timezone_test
