#!/bin/sh
# SPDX-License-Identifier: MIT

. "$(dirname "$0")/common"

list_containers() {
	req_bearer "/api/containers" \
		|| error "Could not list containers"

	[ "$(jq -r '.containers | length' < req)" = 1 ] \
		&& [ "$(jq -r '.containers[0].name' < req)" = alpine ] \
		|| error "API request succeeded but did not find exactly one container up" \
			"Please run this manually: podman kill -a && podman rm -a -f \\" \
			"    && podman run -i -d --init --name alpine docker.io/alpine \\" \
			"        sh -c 'echo ok; exec sleep infinity'"

	check_req_jq image_path '.containers[0].image' docker.io/library/alpine:latest
	check_req_jq running '.containers[0].state' running
	check_req_jq command '.containers[0].command | join(" ")' \
	       	"sh -c echo ok; exec sleep infinity"
	check_req_jq id '.containers[0].id | length' 64
}

containers_stop() {
	req_bearer "/api/containers/alpine/stop" -X POST \
		|| error "Could not stop alpine"
	req_bearer "/api/containers" \
		|| error "Could not list containers"
	check_req_jq exited '.containers[0].state' exited
}

containers_start() {
	req_bearer "/api/containers/alpine/start" -X POST \
		|| error "Could not start alpine"
	req_bearer "/api/containers" \
		|| error "Could not list containers"
	check_req_jq running '.containers[0].state' running
}

containers_logs() {
	req_bearer "/api/containers/alpine/logs" \
		|| error "Could not get alpine logs"
	check_req_jq stdout '.[0].stdout' '"ok\n"' -s
	# should have at least two lines since we just stop/started
	check_req_jq stdout2 '.[1].stdout' '"ok\n"' -s
	check_req_jq exit '.[-1].exit_code' 0 -r -s
}

containers_logs_follow() {
	local pid_kill_remote pid_kill_local marker

	# check we can ssh in, required for this test:
	ssh_armadillo true || skip "Need to be able to ssh into 'armadillo'"
	ssh_armadillo "pgrep -f '^podman logs -f alpine\$'" \
		&& skip "Cannot run if old podman logs were around"

	# use a marker to wait for logs to appear
	marker=$(cat /proc/sys/kernel/random/uuid) \
		|| skip "Could not get random id"
	ssh_armadillo podman exec alpine sh -c "'echo $marker > /proc/1/fd/1'"

	# cannot use req here: curl will be a child of the subshell so we
	# cannot easily just kill that curl
	# hardcode the full command here, we can share it later if required.
	curl -o req -k -s --fail-with-body -H "Authorization: Bearer $(cat token)" --no-buffer \
			"$ABOSWEB_BASE/api/containers/alpine/logs?follow=true" &
	pid_kill_local=$!

	retry grep -q -E "stdout.*$marker" req 2>/dev/null \
		|| error "First marker did not appear: $marker"

	ssh_armadillo podman exec alpine sh -c "'echo $marker > /proc/1/fd/2'"
	retry grep -q -E "stderr.*$marker" req \
		|| error "Second marker did not appear: $marker"

	# Start a second log: we'll kill this podman logs to make sure command
	# ends (reset log to wait for it to really start)
	req_bearer "/api/containers/alpine/logs?follow=true" --no-buffer &
	pid_kill_remote=$!
	retry grep -q -E "stdout.*$marker" req 2>/dev/null \
		|| error "New command didn't show marker: $marker"

	# stop newer podman logs on armadillo, and local old curl here
	ssh_armadillo "pkill -f -n '^podman logs -f alpine\$'"
	kill "$pid_kill_local"
	wait "$pid_kill_local" || true

	retry ssh_armadillo "pgrep -f '^podman logs -f alpine\$'" >/dev/null \
		|| error "podman logs didn't stop on armadillo"

	retry test -e "/proc/$pid_kill_remote" \
		|| error "curl didn't stop when podman logs was killed"
	wait "$pid_kill_remote" \
		|| error "curl failed with non-zero code"
}

# don't bother with other tests if correct container was not running
run_tests list_containers || exit

run_tests containers_stop containers_start containers_logs \
	containers_logs_follow
