// SPDX-License-Identifier: MIT

use anyhow::{Context, Result};
use axum::{
    extract::Path,
    response::IntoResponse,
    routing::{delete, get, post},
    Router,
};
use regex::bytes::Regex;
use serde::{Deserialize, Serialize};
use serde_json::{json, Value};
use std::process::Output;

use crate::common::{
    self, json_response, CheckAuthRestApi, JsonOrForm, RestApiPermissionNetworkAdmin,
    RestApiPermissionNetworkView, RestApiPermissionSmsSend, RestApiPermissionSmsView,
};
use crate::error::ErrorStringResult;

pub fn routes() -> Router {
    Router::new()
        .route("/api/wwan", post(rest_wwan_setup))
        .route("/api/wwan", delete(rest_wwan_delete))
        .route("/api/wwan/sms", post(rest_wwan_send_sms))
        .route("/api/wwan/sms", get(rest_wwan_get_sms_list))
        .route("/api/wwan/sms/{message_id}", get(rest_wwan_get_sms))
        .route("/api/wwan/sms", delete(rest_wwan_delete_sms_all))
        .route("/api/wwan/sms/{message_id}", delete(rest_wwan_delete_sms))
        .route(
            "/api/wwan/signal_quality",
            get(rest_wwan_get_signal_quality),
        )
        .route("/api/wwan/phone_numbers", get(rest_wwan_get_phone_numbers))
        .route("/api/wwan/imei", get(rest_wwan_get_imei))
        .route("/api/wwan/force_restart", post(rest_wwan_force_restart))
}

// connect output is something like this on success (including reconnects,
// which re-use the same id)
//   Setup WWAN connection 'xxxx-xxxx...'
const CONNECT_RE: &str = r#"Setup WWAN connection '([^']*)'"#;

#[derive(Deserialize)]
#[serde(deny_unknown_fields)]
struct WwanSetupParam {
    apn: String,
    user: Option<String>,
    password: Option<String>,
    auth_type: Option<String>,
    mccmnc: Option<String>,
    device: Option<String>,
    ipv6: Option<bool>,
    #[serde(default)]
    no_connect: bool,
}

/// POST "/api/wwan"
/// - Access: Network admin
/// - Input: APN, user, password, auth_type, mccmnc,
///   device, ipv6 (bool)
/// - Output: None
async fn rest_wwan_setup(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkAdmin>,
    JsonOrForm(params): JsonOrForm<WwanSetupParam>,
) -> ErrorStringResult {
    let mut args = vec![
        "wwan_setup.sh",
        common::WWAN_CON_NAME,
        params.device.as_deref().unwrap_or("ttyCommModem"),
        &params.apn,
        params.auth_type.as_deref().unwrap_or("CHAP"),
    ];
    if params.ipv6.unwrap_or(true) {
        args.push("auto")
    } else {
        args.push("ignore")
    }
    if let Some(user) = &params.user {
        args.push("--user");
        args.push(user);
    }
    if let Some(password) = &params.password {
        args.push("--password");
        args.push(password);
    }
    if let Some(mccmnc) = &params.mccmnc {
        args.push("--mccmnc");
        args.push(mccmnc)
    }
    if params.no_connect {
        args.push("--noconnect");
    }
    // build RE before connecting for error handling
    let re = Regex::new(CONNECT_RE).context("Could not build regex?!")?;

    let output = common::exec_command(&args).await?;
    let uuid = re
        .captures(&output.stdout)
        .and_then(|c| c.get(1))
        .context("Connected, but could not get uuid")?;

    json_response(&json!({
        "uuid": String::from_utf8_lossy(uuid.as_bytes()),
    }))
}

/// DELETE "/api/wwan"
/// - Access: Network admin
/// - Input: None
/// - Output: None
async fn rest_wwan_delete(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkAdmin>,
) -> ErrorStringResult {
    common::exec_command(&["wwan_delete.sh", common::WWAN_CON_NAME]).await?;
    Ok(().into_response())
}

#[derive(Deserialize)]
struct SmsMessageParam {
    phone_number: String,
    message: String,
}

async fn get_modem_num() -> Result<String> {
    let args = &["mm-modem-num"];
    match common::exec_command_no_script(args).await {
        Ok(modem_num) => {
            let modem_num_str = String::from_utf8_lossy(&modem_num.stdout);
            Ok(modem_num_str.trim().to_string())
        }
        Err(e) => Err(e),
    }
}

async fn get_modem_json() -> Result<Value> {
    let modem_num = get_modem_num().await?;
    let args = &["mmcli", "-m", &modem_num, "-J"];
    let result = common::exec_command_no_script(args).await?;
    Ok(serde_json::from_str(&String::from_utf8_lossy(
        &result.stdout,
    ))?)
}

async fn create_sms(modem_num: &str, phone_number: &str, message: &str) -> Result<Value> {
    let param = format!(r#"--messaging-create-sms=number={phone_number},text='{message}'"#);
    let args = &["mmcli", "-m", modem_num, &param, "-J"];
    let result = common::exec_command_no_script(args).await?;
    Ok(serde_json::from_str(&String::from_utf8_lossy(
        &result.stdout,
    ))?)
}

async fn send_sms(path: &str) -> Result<Output> {
    let args = &["mmcli", "-s", path, "--send"];
    common::exec_command_no_script(args).await
}

/// POST "/api/wwan/sms"
/// - Access: Sms Send
/// - Input: phone number, message
/// - Output: None
async fn rest_wwan_send_sms(
    _auth: CheckAuthRestApi<RestApiPermissionSmsSend>,
    JsonOrForm(params): JsonOrForm<SmsMessageParam>,
) -> ErrorStringResult {
    let modem_num = get_modem_num().await?;
    let json = create_sms(&modem_num, &params.phone_number, &params.message).await?;
    match &json["modem"]["messaging"]["created-sms"] {
        Value::String(path) => {
            send_sms(path).await?;
            Ok(().into_response())
        }
        _ => Err("mmcli create sms response not formatted as expected")?,
    }
}

async fn sms_list_json(modem_num: &str) -> Result<Value> {
    let args = &["mmcli", "-m", modem_num, "--messaging-list-sms", "-J"];
    let result = common::exec_command_no_script(args).await?;
    Ok(serde_json::from_str(&String::from_utf8_lossy(
        &result.stdout,
    ))?)
}

async fn delete_sms(path: &str) -> Result<Output> {
    let modem_num = get_modem_num().await?;
    let mut param = String::from("--messaging-delete-sms=");
    param.push_str(path);
    let args = &["mmcli", "-m", &modem_num, &param];
    common::exec_command_no_script(args).await
}

/// DELETE "/api/wwan/sms"
/// - Access: Sms View
/// - Input: None
/// - Output: None
async fn rest_wwan_delete_sms_all(
    _auth: CheckAuthRestApi<RestApiPermissionSmsView>,
) -> ErrorStringResult {
    let modem_num = get_modem_num().await?;
    let json = sms_list_json(&modem_num).await?;
    let Value::Array(ref messages) = json["modem.messaging.sms"] else {
        Err("mmcli response badly formatted")?
    };
    for sms in messages {
        let _ = delete_sms(sms.as_str().context("sms path wasn't a string")?).await;
    }
    Ok(().into_response())
}

/// DELETE "/api/wwan/sms/{message_id}"
/// - Access: Sms View
/// - Input: message id
/// - Output: None
async fn rest_wwan_delete_sms(
    _auth: CheckAuthRestApi<RestApiPermissionSmsView>,
    Path(message_id): Path<String>,
) -> ErrorStringResult {
    delete_sms(&message_id).await?;
    Ok(().into_response())
}

#[derive(Serialize, Debug)]
struct SmsList<'a> {
    message_ids: Vec<&'a str>,
}

/// GET "/api/wwan/sms"
/// - Access: Sms View
/// - Input: None
/// - Output: object with sms, list of strings
async fn rest_wwan_get_sms_list(
    _auth: CheckAuthRestApi<RestApiPermissionSmsView>,
) -> ErrorStringResult {
    let modem_num = get_modem_num().await?;
    let json = sms_list_json(&modem_num).await?;
    let Value::Array(ref messages) = json["modem.messaging.sms"] else {
        Err("mmcli response badly formatted")?
    };
    let ids = messages
        .iter()
        .filter_map(|s| Some(s.as_str()?.rsplit_once('/')?.1))
        .collect();
    let list_json = SmsList { message_ids: ids };
    json_response(&list_json)
}

async fn sms_message_json(message_id: &str) -> Result<Value> {
    let args = &["mmcli", "-s", message_id, "-J"];
    let result = common::exec_command_no_script(args).await?;
    Ok(serde_json::from_str(&String::from_utf8_lossy(
        &result.stdout,
    ))?)
}

#[derive(Serialize, Debug)]
struct SmsContent<'a> {
    data: &'a str,
    number: &'a str,
    text: &'a str,
}

#[derive(Serialize, Debug)]
struct SmsProperties<'a> {
    pdu_type: &'a str,
    state: &'a str,
    storage: &'a str,
    timestamp: &'a str,
}

#[derive(Serialize, Debug)]
struct Sms<'a> {
    content: SmsContent<'a>,
    dbus_path: &'a str,
    properties: SmsProperties<'a>,
}

fn mmcli_sms_json_to_str(v: &Value) -> &str {
    v.as_str().unwrap_or("--")
}

/// GET "/api/wwan/sms/{message_id}"
/// - Access: Sms View
/// - Input: None
/// - Output: object with "modem.messaging.sms"
async fn rest_wwan_get_sms(
    _auth: CheckAuthRestApi<RestApiPermissionSmsView>,
    Path(message_id): Path<String>,
) -> ErrorStringResult {
    let mmcli_json = sms_message_json(&message_id).await?;
    let output_json = Sms {
        content: SmsContent {
            data: mmcli_sms_json_to_str(&mmcli_json["sms"]["content"]["data"]),
            number: mmcli_sms_json_to_str(&mmcli_json["sms"]["content"]["number"]),
            text: mmcli_sms_json_to_str(&mmcli_json["sms"]["content"]["text"]),
        },
        dbus_path: mmcli_sms_json_to_str(&mmcli_json["sms"]["dbus-path"]),
        properties: SmsProperties {
            pdu_type: mmcli_sms_json_to_str(&mmcli_json["sms"]["properties"]["pdu-type"]),
            state: mmcli_sms_json_to_str(&mmcli_json["sms"]["properties"]["state"]),
            storage: mmcli_sms_json_to_str(&mmcli_json["sms"]["properties"]["storage"]),
            timestamp: mmcli_sms_json_to_str(&mmcli_json["sms"]["properties"]["timestamp"]),
        },
    };

    json_response(&json!({ "sms": output_json }))
}

/// GET "/api/wwan/signal_quality"
/// - Access: Network View
/// - Input: None
/// - Output: signal_quality
async fn rest_wwan_get_signal_quality(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkView>,
) -> ErrorStringResult {
    let output = common::exec_command(&["wwan_get_signal_quality.sh"]).await?;
    json_response(&json!({
        "signal_quality": String::from_utf8_lossy(&output.stdout)
    }))
}

/// GET "/api/wwan/imei"
/// - Access: Network View
/// - Input: None
/// - Output: imei
async fn rest_wwan_get_imei(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkView>,
) -> ErrorStringResult {
    let json = get_modem_json().await?;
    match &json["modem"]["generic"]["equipment-identifier"] {
        Value::String(imei) => json_response(&json!({ "imei": imei })),
        _ => json_response(&json!({ "imei": "" })),
    }
}

/// GET "/api/wwan/phone_numbers"
/// - Access: Network View
/// - Input: None
/// - Output: object with phone_numbers, list of strings
async fn rest_wwan_get_phone_numbers(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkView>,
) -> ErrorStringResult {
    let json = get_modem_json().await?;
    match &json["modem"]["generic"]["own-numbers"] {
        Value::Array(phone_numbers) => json_response(&json!({ "phone_numbers": phone_numbers })),
        _ => json_response(&json!({ "phone_numbers": "" })),
    }
}

/// POST "/api/wwan/force_restart"
/// - Access: Network Admin
/// - Input: None
/// - Output: None
async fn rest_wwan_force_restart(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkAdmin>,
) -> ErrorStringResult {
    common::exec_command_with_opts(
        &["/usr/bin/wwan-force-restart"],
        &common::CommandOpts {
            is_script: false,
            as_root: true,
            ..Default::default()
        },
    )
    .await?;
    Ok(().into_response())
}
