// SPDX-License-Identifier: MIT

use askama::Template;
use axum::{
    extract::{Extension, Form},
    middleware,
    response::{IntoResponse, Redirect},
    routing::{get, post},
    Router,
};
use serde::Deserialize;
use std::collections::HashMap;

use crate::common::{
    self, check_auth, get_title, networkmanager::nmcon_act, Config, HtmlTemplate, Title,
};
use crate::error::PageResult;

#[cfg(feature = "restapi")]
mod restapi;
#[cfg(not(feature = "restapi"))]
mod restapi {
    pub fn routes() -> axum::Router {
        axum::Router::new()
    }
}

#[derive(Deserialize)]
struct WwanParam {
    apn: String,
    username: String,
    password: Option<String>,
    auth_type: String,
    device: String,
    mccmnc: String,
    ipv6: String,
}

#[derive(Deserialize, Debug)]
struct WwanSaved {
    con: String,
    apn: String,
    user: String,
}

#[derive(Template)]
#[template(path = "../src/wwan/templates/wwan.html")]
struct WwanTemplate {
    wwan_info: HashMap<&'static str, String>,
    wwan_addr: HashMap<&'static str, String>,
    wwan_saved: Vec<WwanSaved>,
    has_wwan: bool,
}

pub fn routes() -> Router {
    Router::new()
        .route("/wwan", get(wwan))
        .route("/wwan_setup", post(wwan_setup))
        .route("/wwan_delete", post(wwan_delete))
        .route("/wwan_connect", post(wwan_connect))
        .route("/wwan_disconnect", post(wwan_disconnect))
        .route_layer(middleware::from_fn(check_auth))
        .merge(restapi::routes())
        .route_layer(middleware::from_fn(|request, next| {
            get_title(request, next, "./wwan")
        }))
}

async fn wwan(Extension(title): Extension<Title>) -> PageResult {
    let interface = common::wwan_interface().unwrap_or("--");
    let wwan_addr = common::current_address("ttyCommModem", interface).await?;
    let wwan_info = common::wwan_info(common::WWAN_CON_NAME).await?;

    let args = &["get_saved_wwan.sh"];
    let output = common::exec_command(args).await?;
    let stdout = String::from_utf8_lossy(&output.stdout);
    let wwan_saved = stdout
        .lines()
        .map(|s| s.trim().split(';').collect::<Vec<&str>>())
        .map(|a| WwanSaved {
            con: a.first().unwrap_or(&"--").to_string(),
            apn: a.get(1).unwrap_or(&"--").to_string(),
            user: a.get(2).unwrap_or(&"--").to_string(),
        })
        .collect();

    let template = WwanTemplate {
        wwan_addr,
        wwan_info,
        wwan_saved,
        has_wwan: Config::get().hardware.has_wwan,
    };
    Ok(HtmlTemplate::new(title.0, template).into_response())
}

async fn wwan_setup(Form(wwan_param): Form<WwanParam>) -> PageResult {
    let mut args = vec![
        "wwan_setup.sh",
        common::WWAN_CON_NAME,
        &wwan_param.device,
        &wwan_param.apn,
        &wwan_param.auth_type,
        &wwan_param.ipv6,
    ];

    if !wwan_param.username.is_empty() {
        args.push("--user");
        args.push(&wwan_param.username);
    }

    if let Some(password) = wwan_param.password.as_ref() {
        args.push("--password");
        args.push(password);
    }

    if !wwan_param.mccmnc.is_empty() {
        args.push("--mccmnc");
        args.push(&wwan_param.mccmnc);
    }

    common::exec_command(&args).await?;

    Ok(Redirect::to("/wwan").into_response())
}

async fn wwan_delete() -> PageResult {
    let args = &["wwan_delete.sh", common::WWAN_CON_NAME];
    common::exec_command(args).await?;

    Ok(Redirect::to("/wwan").into_response())
}

async fn wwan_connect() -> PageResult {
    nmcon_act("up", common::WWAN_CON_NAME).await?;

    Ok(Redirect::to("/wwan").into_response())
}

async fn wwan_disconnect() -> PageResult {
    nmcon_act("down", common::WWAN_CON_NAME).await?;

    Ok(Redirect::to("/wwan").into_response())
}
