// SPDX-License-Identifier: MIT

use anyhow::Result;
use std::collections::HashMap;
use std::path::Path;

use crate::common::exec_command;

const HOSTAPD_CONF: &str = "/etc/hostapd/hostapd.conf";

/// hostapd config content
#[derive(Debug, Default)]
pub struct HostapdConfig {
    pub bridge: Option<String>,
    pub channel: Option<String>,
    pub hw_mode: Option<String>,
    pub interface: Option<String>,
    pub ssid: Option<String>,
}

impl HostapdConfig {
    /// parse one line of 'nmcli -t -c no -f name,uuid,type,device,state con'
    fn from_content(output: &str) -> Self {
        let props = output
            .trim()
            .split('\n')
            .filter(|line| !line.starts_with('#'))
            .filter_map(|line| line.split_once('='))
            .collect::<HashMap<&str, &str>>();
        HostapdConfig {
            bridge: props.get("bridge").map(<&str>::to_string),
            channel: props.get("channel").map(<&str>::to_string),
            hw_mode: match props.get("hw_mode") {
                Some(&"a") => Some(&"5GHz"),
                Some(&"g") => Some(&"2.4GHz"),
                other => other,
            }
            .map(<&str>::to_string),
            interface: props.get("interface").map(<&str>::to_string),
            ssid: props.get("ssid").map(<&str>::to_string),
        }
    }

    pub async fn read() -> Result<Self> {
        if !cfg!(debug_assertions) && !Path::new(HOSTAPD_CONF).is_file() {
            return Ok(HostapdConfig::default());
        }
        Ok(HostapdConfig::from_content(&String::from_utf8_lossy(
            &exec_command(&["get_ap_config.sh"]).await?.stdout,
        )))
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_parse_conf() {
        let ap = HostapdConfig::from_content(
            r#"
hw_mode=a
channel=44
ssid=myap
wpa_passphrase=myap_pass
interface=uap0
bridge=br0
wpa_key_mgmt=WPA-PSK
wpa_pairwise=TKIP
rsn_pairwise=CCMP
driver=nl80211
country_code=JP
ctrl_interface=/var/run/hostapd
ctrl_interface_group=0
disassoc_low_ack=1
preamble=1
wmm_enabled=1
macaddr_acl=0
auth_algs=1
ignore_broadcast_ssid=0
wpa=2
ieee80211ac=1
ieee80211ax=1
ieee80211n=1
ieee80211d=1
ieee80211h=1
logger_syslog=-1
logger_syslog_level=2
logger_stdout=-1
logger_stdout_level=2
"#,
        );
        assert_eq!(&ap.hw_mode.unwrap(), "5GHz");
        assert_eq!(&ap.ssid.unwrap(), "myap");
    }
}
