// SPDX-License-Identifier: MIT

use anyhow::{bail, Context, Result};
use tempfile::TempDir;
use tokio::fs;
use tokio::io::AsyncWriteExt;

use crate::common::{
    self,
    networkmanager::{nmcon_act, nmconvec, NMCon},
    Config,
};

pub struct VpnConfig {
    tmp_dir: TempDir,
}

impl VpnConfig {
    pub fn new() -> Result<Self> {
        Ok(VpnConfig {
            tmp_dir: TempDir::new().context("Could not create temporary directory")?,
        })
    }

    pub fn dir(&self) -> Result<&str> {
        self.tmp_dir
            .path()
            .to_str()
            .context("Temporary dir was not valid unicode?!")
    }

    pub async fn save_file(&self, file_name: &str, data: &[u8]) -> Result<()> {
        let path = self.tmp_dir.path().join(file_name);
        let mut file = fs::File::create(path).await?;
        file.write_all(data).await?;
        Ok(())
    }
}

#[derive(PartialEq)]
pub enum VpnType {
    None,
    Openvpn,
    Wireguard,
}

pub async fn vpn_connection_type() -> VpnType {
    if fs::try_exists("/etc/atmark/abos_web/openvpn/vpn_info.txt")
        .await
        .unwrap_or(false)
    {
        VpnType::Openvpn
    } else if fs::try_exists("/etc/atmark/abos_web/wireguard/vpn_info.txt")
        .await
        .unwrap_or(false)
    {
        VpnType::Wireguard
    } else {
        VpnType::None
    }
}

pub async fn vpn_act(connect: bool) -> Result<()> {
    match vpn_connection_type().await {
        VpnType::Openvpn => {
            if Config::get().software.is_ovpn_installed {
                let cons: Vec<NMCon> = nmconvec()
                    .await?
                    .into_iter()
                    .filter(|conn| &conn.ctype == "vpn")
                    .collect();
                if cons.is_empty() {
                    bail!("No vpn connection found");
                }
                if cons.len() > 1 {
                    bail!("Multiple vpn connections found");
                }
                let act = if connect { "up" } else { "down" };
                nmcon_act(act, &cons[0].uuid).await?;
            } else {
                let act = if connect { "start" } else { "stop" };
                let args = &["podman.sh", act, common::OPENVPN_CONTAINER];
                common::exec_command(args).await?;
            }
        }
        VpnType::Wireguard => {
            let act = if connect { "up" } else { "down" };
            nmcon_act(act, "wg").await?;
        }
        _ => (),
    }

    Ok(())
}
