// SPDX-License-Identifier: MIT

use anyhow::{anyhow, Context, Result};
use serde::Deserialize;
use std::path::{Path, PathBuf};
use tokio::fs;

use crate::common;

const ZONEINFO: &str = "/usr/share/zoneinfo";
const LOCALTIME: &str = "/etc/localtime";

// rust cannot have async functions naively recurse without hiding
// the future they return in a box; async_recursion does that for us.
#[async_recursion::async_recursion]
async fn add_directory_zones(base: &Path, prefix: &Path, zones: &mut Vec<String>) -> Result<()> {
    let dir_path = base.join(prefix);
    let mut dir = match fs::read_dir(dir_path).await {
        Ok(dir) => dir,
        // No such directory:
        // - for root directory it means timezones are not installed, return no timezones
        // - otherwise it should never happen, propagate the error.
        Err(e)
            if e.kind() == std::io::ErrorKind::NotFound
                && prefix.to_str().unwrap_or("not_empty").is_empty() =>
        {
            return Ok(())
        }
        Err(e) => Err(e)?,
    };
    while let Some(dirent) = dir.next_entry().await? {
        let file_name = dirent.file_name();
        let file_name_str = file_name
            .to_str()
            .with_context(|| format!("bad file name: {prefix:?}/{file_name:?}"))?;
        // zoneinfo directory has a few files with dots that are not timezones, skip them
        if file_name_str.contains('.') {
            continue;
        }
        let zone_name = prefix.join(file_name);
        // recurse into subdirs
        if dirent.file_type().await?.is_dir() {
            add_directory_zones(base, &zone_name, zones).await?;
            continue;
        }
        zones.push(
            zone_name
                .into_os_string()
                .into_string()
                .map_err(|ospath| anyhow!("zoneinfo had some non-utf8 file names: {:?}", ospath))?,
        );
    }
    Ok(())
}

pub async fn list_timezones() -> Result<Vec<String>> {
    let mut zones = vec![];
    add_directory_zones(&PathBuf::from(ZONEINFO), &PathBuf::new(), &mut zones).await?;
    zones.sort_unstable();
    Ok(zones)
}

pub async fn get_timezone() -> Result<String> {
    let link = fs::read_link(LOCALTIME).await?;
    let link_str = link
        .to_str()
        .with_context(|| format!("/etc/localtime was not utf8: {link:?}"))?;
    // try a few "common" prefixes and give up
    if let Some(tz) = link_str.strip_prefix("/usr/share/zoneinfo/") {
        return Ok(tz.to_string());
    };
    if let Some(tz) = link_str.strip_prefix("zoneinfo/") {
        return Ok(tz.to_string());
    };
    Ok(link_str.to_string())
}

#[derive(Deserialize)]
pub struct TimezoneParam {
    timezone: String,
}

pub async fn set_timezone(param: TimezoneParam) -> Result<()> {
    common::exec_command(&["time_setup.sh", "timezone", &param.timezone]).await?;
    Ok(())
}

#[cfg(test)]
mod tests {
    use super::*;

    #[tokio::test]
    async fn get_tz() {
        let mut zones = vec![];
        add_directory_zones(
            &PathBuf::from("tests/data/time"),
            &PathBuf::new(),
            &mut zones,
        )
        .await
        .unwrap();
        zones.sort();
        assert_eq!(&zones, &["subdir/tz2", "tz1"])
    }
}
