// SPDX-License-Identifier: MIT

use anyhow::{anyhow, Result};
use serde::{Deserialize, Serialize};
use tokio::fs;
use tracing::warn;

use crate::common;

const SERVER_PATHS: [&str; 3] = [
    "/etc/atmark/chrony.conf.d/servers.conf",
    "/etc/chrony/conf.d/servers.conf",
    "/lib/chrony.conf.d/servers.conf",
];
const INITSTEPSLEW_PATHS: [&str; 3] = [
    "/etc/atmark/chrony.conf.d/initstepslew.conf",
    "/etc/chrony/conf.d/initstepslew.conf",
    "/lib/chrony.conf.d/initstepslew.conf",
];

async fn get_from_files(paths: &[&str]) -> Option<Vec<String>> {
    for path in paths {
        let Ok(content) = fs::read_to_string(path).await else {
            continue;
        };
        return Some(
            content
                .split('\n')
                .filter(|line| !line.starts_with('#') && !line.is_empty())
                .map(String::from)
                .collect(),
        );
    }
    None
}

pub async fn get_servers() -> Vec<String> {
    get_from_files(&SERVER_PATHS).await.unwrap_or_else(|| {
        warn!("No chrony servers.conf config found!");
        vec![]
    })
}

pub async fn get_initstepslew() -> Option<String> {
    get_from_files(&INITSTEPSLEW_PATHS)
        .await
        .unwrap_or_else(|| {
            warn!("No chrony initstepslew.conf config found!");
            vec![]
        })
        .iter()
        .filter_map(|line| line.strip_prefix("initstepslew "))
        .next()
        .map(|s| s.to_string())
}

#[derive(Serialize)]
pub struct NtpSettings {
    pub servers: Vec<String>,
    pub initstepslew: Option<String>,
}

pub async fn get_config() -> Result<NtpSettings> {
    // run both in parallel
    let (servers, initstepslew) = futures::join!(get_servers(), get_initstepslew());
    Ok(NtpSettings {
        servers,
        initstepslew,
    })
}

#[derive(Deserialize)]
pub struct NtpSettingsParam {
    #[serde(default)]
    pub servers: Vec<String>,
    // using a vec here allows differentiating between no argument (empty vec) and empty argument
    // (vec with empty string inside)
    #[serde(default)]
    pub initstepslew: Vec<String>,
}

pub async fn set_config(mut settings: NtpSettingsParam) -> Result<()> {
    let mut args = vec!["ntp_setup.sh".to_string()];
    if !settings.servers.is_empty() {
        args.push("source".to_string());
        args.push(settings.servers.join("\n"))
    }
    match settings.initstepslew.len() {
        0 => (),
        1 => {
            args.push("initstepslew".to_string());
            args.push(settings.initstepslew.remove(0));
        }
        _ => return Err(anyhow!("Cannot have multiple initstepslew")),
    };
    if args.len() == 1 {
        return Err(anyhow!("Nothing to set"));
    }
    common::exec_command(&args).await?;
    Ok(())
}

pub struct NtpStatus {
    pub server_ip: String,
    pub offset: String,
}

pub async fn get_info() -> Result<Option<NtpStatus>> {
    // 'chronyc -c tracking' outputs something as follow:
    // 85F3EEF3,133.243.238.243,2,1709882844.682764904,-0.000009062,0.000007834,0.000059569,-10.196,0.002,0.197,0.017258352,0.000112419,65.2,Normal
    // 00000000,,0,0.000000000,0.000000000,0.000000000,0.000000000,-9.131,0.000,0.000,1.000000000,1.000000000,0.0,Not synchronised
    // - [0] reference id
    // - [1] IP of server currently tracked
    // - [2] stratum of server
    // - [3] "ref time"
    // - [4] difference with system time
    //   a negative value here means system is fast of ntp
    // - [5] last offset
    // - [6] RMS offset
    // - [7] frequency
    // - [8] residual freq
    // - [9] skew
    // - [10] root delay
    // - [11] root dispertion
    // - [12] update interval
    // - [13] leap status
    let status = common::exec_command(&["ntp_info.sh"]).await?;
    let output = String::from_utf8_lossy(&status.stdout);
    let values: Vec<&str> = output.split(',').collect();
    if values.get(1).unwrap_or(&"").is_empty() {
        return Ok(None);
    }
    Ok(Some(NtpStatus {
        server_ip: values.get(1).unwrap_or(&"").to_string(),
        offset: values.get(4).unwrap_or(&"").to_string(),
    }))
}

#[cfg(test)]
mod tests {
    use super::*;

    #[tokio::test]
    async fn get_servers() {
        assert_eq!(get_from_files(&["no_such_file"]).await, None);
        assert_eq!(
            get_from_files(&["no_such_file", "tests/data/time/servers1.conf"])
                .await
                .unwrap(),
            vec!["server example.com iburst"]
        );
        assert_eq!(
            get_from_files(&[
                "tests/data/time/servers2.conf",
                "tests/data/time/servers1.conf"
            ])
            .await
            .unwrap(),
            vec!["pool pool.ntp.org iburst", "server example.com iburst"]
        );
    }
}
