// SPDX-License-Identifier: MIT

use anyhow::{Context, Result};
use chrono::{DateTime, Local};
use std::cmp::Ordering;
use tokio::fs;

use crate::common::SliceExt;

#[derive(Debug, PartialEq)]
pub struct SwuVersion {
    pub component: String,
    pub version: String,
}

#[derive(Debug, PartialEq)]
pub struct SwuVersionUpdate {
    pub component: String,
    pub old_version: Option<String>,
    pub new_version: Option<String>,
}
pub struct SwuLastUpdate {
    pub rootfs: String,
    pub updated_rootfs: String,
    pub updated_date: DateTime<Local>,
    pub updated_versions: Vec<SwuVersionUpdate>,
}

pub async fn get_swu_versions() -> Result<Vec<SwuVersion>> {
    fs::read("/etc/sw-versions")
        .await
        .context("Could not read /etc/sw-versions")
        .map(|s| parse_swu_versions(&s))
}

fn get_rootfs_from_cmdline(cmdline: &str) -> Option<String> {
    cmdline
        .split(' ')
        .find_map(|s| s.strip_prefix("root=/dev/"))
        .map(|s| s.to_string())
}

fn get_other_rootfs(rootfs: &str) -> Option<String> {
    let index = rootfs.chars().next_back()?;
    let other_index = match index {
        '1' => '2',
        '2' => '1',
        _ => return None,
    };
    let mut other_rootfs = String::with_capacity(rootfs.len());
    other_rootfs.push_str(&rootfs[0..rootfs.len() - 1]);
    other_rootfs.push(other_index);
    Some(other_rootfs)
}

fn compare_versions(
    old_versions: &[SwuVersion],
    new_versions: &[SwuVersion],
) -> Vec<SwuVersionUpdate> {
    // versions are guaranteed to be sorted,
    // so just process in order
    let mut old_iter = old_versions.iter();
    let mut new_iter = new_versions.iter();
    let mut old_option = old_iter.next();
    let mut new_option = new_iter.next();
    let mut diff = vec![];
    while old_option.is_some() && new_option.is_some() {
        let old = old_option.unwrap();
        let new = new_option.unwrap();
        match old.component.cmp(&new.component) {
            Ordering::Less => {
                diff.push(SwuVersionUpdate {
                    component: old.component.clone(),
                    old_version: Some(old.version.clone()),
                    new_version: None,
                });
                old_option = old_iter.next()
            }
            Ordering::Greater => {
                diff.push(SwuVersionUpdate {
                    component: new.component.clone(),
                    old_version: None,
                    new_version: Some(new.version.clone()),
                });
                new_option = new_iter.next()
            }
            Ordering::Equal => {
                if old.version != new.version {
                    diff.push(SwuVersionUpdate {
                        component: new.component.clone(),
                        old_version: Some(old.version.clone()),
                        new_version: Some(new.version.clone()),
                    });
                };
                new_option = new_iter.next();
                old_option = old_iter.next();
            }
        }
    }
    diff
}

pub async fn get_last_update(current_versions: &[SwuVersion]) -> Option<SwuLastUpdate> {
    let last_update = fs::read_to_string("/var/log/swupdate/last_update")
        .await
        .ok()?;
    let (updated_rootfs, updated_timestamp) = last_update.trim().split_once(' ')?;
    let updated_rootfs = updated_rootfs
        .strip_prefix("/dev/")
        .unwrap_or(updated_rootfs);
    let updated_date = DateTime::from_timestamp(updated_timestamp.parse().ok()?, 0)?.into();
    let cmdline = fs::read_to_string("/proc/cmdline").await.ok()?;
    let rootfs = get_rootfs_from_cmdline(&cmdline)?;
    let other_rootfs = get_other_rootfs(&rootfs)?;
    let old_versions_file = format!("/var/log/swupdate/sw-versions-{other_rootfs}");
    let updated_versions = match fs::read(old_versions_file).await {
        Ok(content) => {
            let old_versions = parse_swu_versions(&content);
            compare_versions(&old_versions, current_versions)
        }
        _ => vec![],
    };
    Some(SwuLastUpdate {
        rootfs,
        updated_rootfs: updated_rootfs.into(),
        updated_date,
        updated_versions,
    })
}

/// just split by line and space
fn parse_swu_versions(content: &[u8]) -> Vec<SwuVersion> {
    let mut versions = content
        .split(|c| *c == b'\n')
        .map(|s| s.trim())
        .filter_map(|s| {
            // Switch to s.split_once() once stabilized
            let (component, version) = SliceExt::split_once(s, |b| *b == b' ')?;
            Some(SwuVersion {
                component: String::from_utf8_lossy(component).into(),
                version: String::from_utf8_lossy(version).into(),
            })
        })
        .collect::<Vec<SwuVersion>>();

    versions.sort_unstable_by(|a, b| a.component.cmp(&b.component));
    versions
}

#[cfg(test)]
mod tests {
    use crate::swu::version::*;

    #[test]
    fn test_parse_swu_versions() {
        let versions = parse_swu_versions(
            br#"test 1
 test2 2 with space"#,
        );
        assert_eq!(
            &versions,
            &[
                SwuVersion {
                    component: "test".into(),
                    version: "1".into()
                },
                SwuVersion {
                    component: "test2".into(),
                    version: "2 with space".into()
                },
            ]
        );
        let versions = parse_swu_versions(b"");
        assert_eq!(versions, &[]);
    }

    #[test]
    fn test_compare_versions() {
        let versions1 = parse_swu_versions(
            br#"common 1
one 1
updated 1"#,
        );
        let mut second_version_with_invalid_utf8 = br#"common 1
two 1
updated 2
junk a"#
            .to_owned();
        second_version_with_invalid_utf8[second_version_with_invalid_utf8.len() - 1] = 0xfd;
        let versions2 = parse_swu_versions(&second_version_with_invalid_utf8);
        let diff = compare_versions(&versions1, &versions2);
        assert_eq!(
            &diff,
            &[
                SwuVersionUpdate {
                    component: "junk".into(),
                    old_version: None,
                    new_version: Some("�".into())
                },
                SwuVersionUpdate {
                    component: "one".into(),
                    old_version: Some("1".into()),
                    new_version: None
                },
                SwuVersionUpdate {
                    component: "two".into(),
                    old_version: None,
                    new_version: Some("1".into())
                },
                SwuVersionUpdate {
                    component: "updated".into(),
                    old_version: Some("1".into()),
                    new_version: Some("2".into())
                }
            ]
        );
    }

    #[test]
    fn test_rootfs() {
        let rootfs = get_rootfs_from_cmdline(
            "console=ttymxc1,115200 root=/dev/mmcblk2p2 rootwait ro quiet nokaslr",
        )
        .unwrap();
        assert_eq!(&rootfs, "mmcblk2p2");
        assert!(get_rootfs_from_cmdline("test").is_none());
        let other_rootfs = get_other_rootfs(&rootfs).unwrap();
        assert_eq!(&other_rootfs, "mmcblk2p1");
        assert_eq!(other_rootfs.capacity(), other_rootfs.len());
        assert!(get_other_rootfs("mmcblk2p3").is_none());
        assert!(get_other_rootfs("").is_none());
    }
}
