// SPDX-License-Identifier: MIT

use askama::Template;
use axum::{
    extract::{
        ws::{self, WebSocket, WebSocketUpgrade},
        Extension,
    },
    http::StatusCode,
    middleware,
    response::IntoResponse,
    routing::get,
    Router,
};
use futures::{sink::SinkExt, stream::StreamExt};
use tower_sessions::Session;
use tracing::{trace, warn};

use crate::common::{
    check_auth, exec_command, filters, get_title, stream_command, stream_command_input,
    CommandOpts, Config, HtmlTemplate, LoggedIn, Title,
};
use crate::error::{ErrorStringResult, PageResult};

mod version;
pub use version::{get_last_update, get_swu_versions, SwuLastUpdate, SwuVersion};

#[cfg(feature = "restapi")]
mod restapi;
#[cfg(not(feature = "restapi"))]
mod restapi {
    pub fn routes() -> axum::Router {
        axum::Router::new()
    }
}

#[derive(Clone)]
struct SwuUrlOption<'a> {
    url: &'a str,
    display_name: &'a str,
}

impl<'a> SwuUrlOption<'a> {
    fn from_url(url: &'a str) -> Self {
        let display_name = url.split('/').next_back().unwrap_or(url);
        Self { url, display_name }
    }
}

impl std::fmt::Display for SwuUrlOption<'_> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}", self.display_name)
    }
}

#[derive(Template)]
#[template(path = "../src/swu/templates/swu.html")]
struct SwuTemplate<'a> {
    initial_setup_installed: bool,
    swupdate_watch_url: String,
    swupdate_watch_urls: Vec<SwuUrlOption<'a>>,
    swu_versions: Vec<SwuVersion>,
    swu_last_update: Option<SwuLastUpdate>,
    update_url: Option<String>,
}

fn get_update_url(product_name: &str) -> Option<String> {
    let trimmed = product_name.trim().trim_matches('"');
    let base_url = "https://armadillo.atmark-techno.com";
    let suffix = "/news/software-updates?keyword=Armadillo+Base+OS";

    let armadillo_path = match trimmed {
        "Armadillo-IoT A9E" => "armadillo-iot-a9e",
        "Armadillo-900" => "armadillo-900",
        "Armadillo-640" => "armadillo-640",
        "Armadillo-610" => "armadillo-610",
        "Armadillo-IoT A6E" => "armadillo-iot-a6e",
        "Armadillo-IoT G4" => "armadillo-iot-g4",
        "high-g1-es1" => "armadillo-iot-g4",
        "Armadillo-X2" => "armadillo-x2",
        _ => return None,
    };

    Some(format!("{base_url}/{armadillo_path}{suffix}"))
}

pub fn routes() -> Router {
    Router::new()
        .route("/swu", get(swu))
        .route("/swu_ws", get(swu_ws))
        .route_layer(middleware::from_fn(check_auth))
        .merge(restapi::routes())
        .route_layer(middleware::from_fn(|request, next| {
            get_title(request, next, "./swu")
        }))
}

async fn swu(Extension(title): Extension<Title>) -> PageResult {
    let swu_versions = get_swu_versions().await.unwrap_or_default();
    let swu_last_update = get_last_update(&swu_versions).await;
    let update_url = get_update_url(&Config::get().hardware.product);
    let swu_config = &Config::get().swu;

    let template = SwuTemplate {
        initial_setup_installed: swu_config.initial_setup_installed,
        swupdate_watch_url: swu_config.watch_urls.join(" "),
        swupdate_watch_urls: swu_config
            .watch_urls
            .iter()
            .map(|url| SwuUrlOption::from_url(url))
            .collect(),
        swu_versions,
        swu_last_update,
        update_url,
    };
    Ok(HtmlTemplate::new(title.0, template).into_response())
}

async fn swu_ws(session: Session, ws: WebSocketUpgrade) -> ErrorStringResult {
    if !cfg!(debug_assertions) && !session.logged_in().await {
        // close socket immediately
        Err((StatusCode::UNAUTHORIZED, "not logged in"))?;
    }
    Ok(ws.on_upgrade(swu_socket_handler).into_response())
}

async fn swu_socket_handler(mut socket: WebSocket) {
    // websocket "protocol"
    // first message should be Text() with a keyword:
    //  * "file": second message onwards are binary file content
    //  * "url": second message onwards are url (space separated), terminated by "END"
    //  * "log": no more message accepted
    let Some(Ok(ws::Message::Text(command))) = socket.recv().await else {
        warn!("Socket first message was not text");
        return;
    };
    trace!("Got {} on websocket", command);
    match command.as_str() {
        "file" => swu_file(&mut socket).await,
        "url" => swu_url(&mut socket).await,
        "latest_log" => swu_latest_log(&mut socket).await,
        _ => warn!("Invalid command on websocket: {}", command),
    }
    // wait for close from client: if we close first,
    // the browser will not be able to process all the messages until the end
    let _ = socket.send(ws::Message::Close(None)).await;
    while let Some(Ok(_)) = socket.recv().await {}
}

async fn swu_file(socket: &mut WebSocket) {
    let (mut output, mut input) = socket.split();
    if let Err(e) = stream_command_input(
        &["swupdate.sh", "stdin"],
        &CommandOpts {
            stream_ignore_output_errors: true,
            ..Default::default()
        },
        Some(&mut input),
        &mut output,
    )
    .await
    {
        warn!("Error streaming swupdate file: {:?}", e);
        return;
    };
    let _ = output
        .send(ws::Message::text("swupdate exited".to_string()))
        .await;
}

async fn swu_url(socket: &mut WebSocket) {
    let (mut output, mut input) = socket.split();
    let mut cmd = vec!["swupdate.sh".to_string(), "url".to_string()];
    'outer: while let Some(Ok(ws::Message::Text(url))) = input.next().await {
        for u in url.trim().split(' ') {
            match u {
                "END" => break 'outer,
                _ => cmd.push(u.to_string()),
            }
        }
    }
    if cmd.len() == 2 {
        let _ = output
            .send(ws::Message::text("No URL given?".to_string()))
            .await;
        return;
    }
    trace!("Starting {:?}", cmd);
    if let Err(e) = stream_command(
        &cmd,
        &CommandOpts {
            stream_ignore_output_errors: true,
            ..Default::default()
        },
        &mut output,
    )
    .await
    {
        warn!("Error streaming swupdate url: {:?}", e);
        return;
    };
    let _ = output
        .send(ws::Message::text("swupdate exited".to_string()))
        .await;
}
async fn swu_latest_log(socket: &mut WebSocket) {
    let log = match exec_command(&["get_latest_swupdate_log.sh"]).await {
        Ok(status) => String::from_utf8_lossy(&status.stdout).to_string(),
        Err(_) => "ログを取得できませんでした".to_string(),
    };
    let _ = socket.send(ws::Message::text(log)).await;
}
