// SPDX-License-Identifier: MIT

use anyhow::Result;
use askama::Template;
use axum::{
    extract::Form,
    response::{IntoResponse, Redirect},
};
use serde::Deserialize;
use tokio::fs;

use crate::common::{self, Config};
use crate::error::PageResult;

#[derive(Template)]
#[template(path = "../src/settings/templates/settings_sshd.html")]
pub struct SettingsSshdTemplate {
    menu_enabled: bool,
    sshd_enabled: bool,
    rootpw_enabled: bool,
}

pub async fn sshd_settings() -> SettingsSshdTemplate {
    SettingsSshdTemplate {
        menu_enabled: Config::get().software.manage_root,
        sshd_enabled: fs::try_exists("/etc/runlevels/default/sshd")
            .await
            .unwrap_or(false),
        rootpw_enabled: fs::try_exists("/etc/ssh/sshd_config.d/50_abos_web.conf")
            .await
            .unwrap_or(false),
    }
}

pub async fn sshd_setup(
    sshd_enable: Option<bool>,
    rootpw_enable: Option<bool>,
    mut sshd_pubkey: Option<String>,
) -> Result<()> {
    let mut args = vec!["sshd_setup.sh"];
    if let Some(enable) = sshd_enable {
        if enable {
            args.push("sshd_enable");
        } else {
            args.push("sshd_disable");
        }
    }
    if let Some(enable) = rootpw_enable {
        if enable {
            args.push("rootpw_enable");
        } else {
            args.push("rootpw_disable");
        }
    }
    if let Some(ref mut pubkey) = &mut sshd_pubkey {
        args.push("sshd_pubkey");
        if !pubkey.ends_with('\n') {
            pubkey.push('\n')
        }
    }
    common::exec_command_stdin(&args, &sshd_pubkey.unwrap_or_default()).await?;
    Ok(())
}

#[derive(Deserialize)]
pub struct SshdSetupParam {
    sshd_current_enabled: bool,
    // browsers do not send anything when checkbox is unchecked, so default false
    #[serde(default)]
    sshd_enable: bool,
    rootpw_current_enabled: bool,
    #[serde(default)]
    rootpw_enable: bool,
    sshd_pubkey: String,
}

pub async fn settings_sshd_setup(Form(param): Form<SshdSetupParam>) -> PageResult {
    let sshd_enable = if param.sshd_enable != param.sshd_current_enabled {
        Some(param.sshd_enable)
    } else {
        None
    };
    let rootpw_enable = if param.rootpw_enable != param.rootpw_current_enabled {
        Some(param.rootpw_enable)
    } else {
        None
    };
    let pubkey = if !param.sshd_pubkey.is_empty() {
        Some(param.sshd_pubkey)
    } else {
        None
    };
    sshd_setup(sshd_enable, rootpw_enable, pubkey).await?;
    Ok(Redirect::to("/settings").into_response())
}
