// SPDX-License-Identifier: MIT

use axum::{
    extract::Path,
    response::IntoResponse,
    routing::{delete, get, post},
    Router,
};
use serde::{Deserialize, Serialize};

use crate::common::{
    json_response, restapi, CheckAuthRestApi, JsonOrFormOption, RestApiPermission,
    RestApiPermissionAdmin, RestApiSerializeToken, RestApiToken, SerializeTokenOptions,
};
use crate::error::ErrorStringResult;

pub fn routes() -> Router {
    Router::new()
        .route("/api/tokens", get(restapi_list))
        .route("/api/tokens", post(restapi_new))
        .route("/api/tokens/{token_id}", get(restapi_get))
        .route("/api/tokens/{token_id}", post(restapi_mod))
        .route("/api/tokens/{token_id}", delete(restapi_delete))
}

#[derive(Serialize)]
pub struct RestApiList<'a> {
    pub tokens: Vec<RestApiSerializeToken<'a>>,
}

/// GET "/api/tokens"
/// - Access: admin
/// - Input: None
/// - Output: json object with 'tokens' key, containing a list of tokens (token, permissions)
async fn restapi_list(_auth: CheckAuthRestApi<RestApiPermissionAdmin>) -> ErrorStringResult {
    let tokens = restapi::list_tokens().await?;
    let tokens_list = RestApiList {
        tokens: tokens
            .iter()
            .map(|t| {
                t.to_serialize(&SerializeTokenOptions {
                    include_token_id: true,
                })
            })
            .collect(),
    };
    json_response(&tokens_list)
}

// We'd normally use RestApiDeserializeToken but for some reason using
// Option<Vec<RestApiPermission>> here makes requests with form and a single item
// (e.g. `curl -d permissions=Admin`) fail with "invalid type: string \"Admin\", expected a
// sequence"
// Removing the option works and does not change behaviour at this point; we'll need to
// figure this out when we add other settings than permissions so individual settings
// can be made optional...
#[derive(Deserialize)]
#[serde(deny_unknown_fields)]
struct PermissionsParam {
    permissions: Vec<RestApiPermission>,
}

/// POST "/api/tokens"
/// - Access: admin
/// - Input: optional permission array (default to admin)
/// - Output: json object with (token, permission) of new token
async fn restapi_new(
    _auth: CheckAuthRestApi<RestApiPermissionAdmin>,
    JsonOrFormOption(param): JsonOrFormOption<PermissionsParam>,
) -> ErrorStringResult {
    // JsonOrFormOption means we accept plain post (no content), json parameters
    // or form (normal curl -d) parameters.
    // The option inside means that if we get json or form, some fields can be missing
    let permissions = param
        .map(|param| param.permissions)
        .unwrap_or_else(|| vec![RestApiPermission::Admin]);
    let token = restapi::create_token(permissions).await?;
    json_response(&token.to_serialize(&SerializeTokenOptions {
        include_token_id: true,
    }))
}

/// GET "/api/tokens/{token_id}"
/// - Access: admin
/// - Input: None
/// - Output: a token (token, permissions) object as in list
async fn restapi_get(
    _auth: CheckAuthRestApi<RestApiPermissionAdmin>,
    Path(token_id): Path<String>,
) -> ErrorStringResult {
    let token = restapi::get_token(&token_id).await?;
    json_response(&token.to_serialize(&SerializeTokenOptions {
        include_token_id: true,
    }))
}

/// POST "/api/tokens/{token_id}"
/// - Access: admin
/// - Input: optional permission array (default to admin)
/// - Output: modified token (token, permissions) object.
///   Token is created if it did not exist.
async fn restapi_mod(
    _auth: CheckAuthRestApi<RestApiPermissionAdmin>,
    Path(token_id): Path<String>,
    JsonOrFormOption(param): JsonOrFormOption<PermissionsParam>,
) -> ErrorStringResult {
    let mut token = restapi::get_token(&token_id).await.unwrap_or(RestApiToken {
        token_id,
        permissions: vec![RestApiPermission::Admin],
    });
    if let Some(permissions) = param.map(|p| p.permissions) {
        token.permissions = permissions;
    }
    restapi::write_token(&token).await?;
    json_response(&token.to_serialize(&SerializeTokenOptions {
        include_token_id: true,
    }))
}

/// DELETE "/api/tokens/{token_id}"
/// - Access: admin
/// - Input: None
/// - Output: None
async fn restapi_delete(
    _auth: CheckAuthRestApi<RestApiPermissionAdmin>,
    Path(token_id): Path<String>,
) -> ErrorStringResult {
    restapi::delete_token(&token_id).await?;
    Ok(().into_response())
}
