// SPDX-License-Identifier: MIT

use anyhow::Result;
use openssl::ssl::{Ssl, SslAcceptor, SslFiletype, SslMethod};
use std::path::PathBuf;
use std::pin::Pin;
use tokio::net::TcpStream;
use tokio_openssl::SslStream;

use crate::server::args;

pub type Acceptor = SslAcceptor;
pub async fn tls_init() -> Result<Acceptor> {
    let args = args();
    let mut tls_builder = SslAcceptor::mozilla_modern_v5(SslMethod::tls())?;
    tls_builder.set_certificate_file(PathBuf::from(&args.tls_cert), SslFiletype::PEM)?;
    tls_builder.set_private_key_file(PathBuf::from(&args.tls_key), SslFiletype::PEM)?;
    tls_builder.check_private_key()?;
    Ok(tls_builder.build())
}

pub async fn tls_accept(acceptor: Acceptor, cnx: TcpStream) -> Result<SslStream<TcpStream>> {
    let ssl = Ssl::new(acceptor.context())?;
    let mut stream = SslStream::new(ssl, cnx)?;
    SslStream::accept(Pin::new(&mut stream)).await?;
    Ok(stream)
}
