// SPDX-License-Identifier: MIT

use anyhow::{Error, Result};
use askama::Template;
use axum::{middleware, response::IntoResponse, routing::get, Router};
use std::collections::HashMap;

use crate::common::{self, check_auth, Config, RawTemplate};
use crate::error::ErrorStringResult;
use crate::status;
use crate::wlan::get_wlan_info;

#[derive(Template)]
#[template(path = "../src/nwstatbar/templates/nwstatbar.html")]
struct NWStatBarTemplate {
    nw_stats: Vec<HashMap<&'static str, String>>,
}

pub fn routes() -> Router {
    Router::new()
        .route("/nwstatbar", get(nwstatbar))
        .route_layer(middleware::from_fn(check_auth))
}

async fn nwstatbar() -> ErrorStringResult {
    let nw_stats = nw_stats().await?;
    let template = NWStatBarTemplate { nw_stats };
    Ok(RawTemplate(template).into_response())
}

async fn current_addr<'a>(
    dev_name: &str,
    interface: &str,
) -> Result<HashMap<&'a str, String>, Error> {
    let addr = match common::current_address(dev_name, interface).await {
        Ok(val) => val,
        Err(e) => return Err(Error::msg(e)),
    };
    Ok(addr)
}

async fn nif_stat(interface: &str) -> Result<String, Error> {
    let addr = current_addr(interface, interface).await?;
    let stat = interface.to_string()
        + " "
        + if addr["current_ip_addr"] == "--" {
            "down"
        } else {
            "up"
        };
    Ok(stat)
}

async fn lan_stat() -> Result<String, Error> {
    let mut stat = nif_stat("eth0").await?;
    if common::exists_eth1() {
        stat = stat + " / " + &nif_stat("eth1").await?;
    };
    Ok(stat)
}

async fn wlan_stat() -> Result<String, Error> {
    let wlan_info = get_wlan_info().await?;
    if wlan_info.hostapd.ssid.is_none() {
        if wlan_info.sta.ssid.is_none() {
            Ok("接続なし".to_string())
        } else {
            Ok("STAモード".to_string())
        }
    } else {
        Ok(format!(
            "APモード/{}",
            wlan_info.hostapd.hw_mode.as_deref().unwrap_or("?")
        ))
    }
}

async fn wwan_stat() -> Result<(String, String), Error> {
    let (wwan_addr, wwan_info) = status::wwan_info().await?;
    let stat = if wwan_addr["current_ip_addr"] == "--" {
        "接続なし"
    } else {
        "接続中"
    };
    let signal_quality = match wwan_info["signalq"].parse::<i8>() {
        Ok(val) => {
            if !(0..=100).contains(&val) {
                -1
            } else {
                match val {
                    0..=39 => 0,
                    40..=59 => 1,
                    60..=79 => 2,
                    _ => 3,
                }
            }
        }
        Err(_e) => -1,
    };
    Ok((stat.to_string(), signal_quality.to_string()))
}

fn push_stat(statvec: &mut Vec<HashMap<&'static str, String>>, stat: String, name: &'static str) {
    let mut map: HashMap<&'static str, String> = HashMap::new();

    map.insert("name", name.to_string());
    map.insert("value", stat);
    statvec.push(map);
}

async fn nw_stats() -> Result<Vec<HashMap<&'static str, String>>, Error> {
    let mut nw_stats: Vec<HashMap<&'static str, String>> = Vec::new();
    let config = Config::get();

    if config.hardware.has_wwan {
        let (stat, sq) = wwan_stat().await?;
        push_stat(&mut nw_stats, stat, "WWAN");
        nw_stats[0].insert("signal_quality", sq);
    }
    if common::wlan_interface().is_some() {
        let stat = wlan_stat().await?;
        push_stat(&mut nw_stats, stat, "WLAN");
    }
    let stat = lan_stat().await?;
    push_stat(&mut nw_stats, stat, "LAN");
    Ok(nw_stats)
}
