// SPDX-License-Identifier: MIT

use askama::Template;
use axum::{
    extract::Form,
    response::{IntoResponse, Redirect},
    routing::{get, post},
    Router,
};
use serde::Deserialize;
use tower_sessions::Session;

use crate::common::{self, check_password, current_password_hash, LoggedIn, RawTemplate};
use crate::error::PageResult;

#[cfg(feature = "restapi")]
mod restapi;
#[cfg(not(feature = "restapi"))]
mod restapi {
    pub fn routes() -> axum::Router {
        axum::Router::new()
    }
}

#[derive(Deserialize, Debug, Clone)]
struct User {
    password: String,
}

#[derive(Template)]
#[template(path = "../src/login/templates/login.html")]
struct LoginTemplate {
    login_error: bool,
    version: &'static str,
}

#[derive(Deserialize, Debug, Clone)]
struct RegisterPasswordParam {
    password: String,
    password_confirm: String,
}

#[derive(Template)]
#[template(path = "../src/login/templates/new_password.html")]
struct NewPasswordTemplate {
    confirm_error: bool,
    first_time: bool,
    length_error: bool,
    version: &'static str,
}

#[derive(Template)]
#[template(path = "../src/login/templates/success.html")]
struct SuccessTemplate {
    version: &'static str,
}

pub fn routes() -> Router {
    // these routes all check for being logged in manually if required
    Router::new()
        .route("/", get(index))
        .route("/login", post(login))
        .route("/logout", get(logout))
        .route("/new_password", get(new_password))
        .route("/register_password", post(register_password))
        .route("/success", get(success))
        .merge(restapi::routes())
}

async fn index(session: Session) -> PageResult {
    if session.logged_in().await {
        return Ok(Redirect::to("/top").into_response());
    }
    if !cfg!(debug_assertions) {
        let hash = current_password_hash().await?;
        if hash == "!" {
            return Ok(Redirect::to("/new_password").into_response());
        }
    }

    let login_error = session.logged_in().await;
    session.remove::<bool>("login_error").await?;
    let template = LoginTemplate {
        login_error,
        version: common::VERSION,
    };
    Ok(RawTemplate(template).into_response())
}

async fn login(session: Session, Form(user): Form<User>) -> PageResult {
    let password = user.password;
    if cfg!(debug_assertions) {
        if password.is_empty() {
            return Ok(Redirect::to("/new_password").into_response());
        }
        if session.insert("login", true).await.is_err() {
            return Ok(Redirect::to("/").into_response());
        }
        Ok(Redirect::to("/top").into_response())
    } else if check_password(&password).await.is_ok() {
        if session.insert("login", true).await.is_err() {
            return Ok(Redirect::to("/").into_response());
        }
        Ok(Redirect::to("/top").into_response())
    } else {
        let _ = session.insert("login_error", true).await;
        Ok(Redirect::to("/").into_response())
    }
}

async fn logout(session: Session) -> PageResult {
    session.flush().await?;
    Ok(Redirect::to("/").into_response())
}

async fn new_password(session: Session) -> PageResult {
    let first_time = if session.logged_in().await {
        false
    } else if current_password_hash().await? == "!" {
        true
    } else {
        return Ok(Redirect::to("/").into_response());
    };
    let confirm_error = session
        .remove::<bool>("confirm_error")
        .await
        .unwrap_or(Some(false))
        .unwrap_or(false);
    let length_error = session
        .remove::<bool>("length_error")
        .await
        .unwrap_or(Some(false))
        .unwrap_or(false);
    let template = NewPasswordTemplate {
        confirm_error,
        first_time,
        length_error,
        version: common::VERSION,
    };
    Ok(RawTemplate(template).into_response())
}

async fn register_password(
    session: Session,
    Form(password_param): Form<RegisterPasswordParam>,
) -> PageResult {
    if !session.logged_in().await && current_password_hash().await? != "!" {
        return Ok(Redirect::to("/").into_response());
    }

    if password_param.password.chars().count() < 8 {
        session.insert("length_error", true).await?;
        return Ok(Redirect::to("/new_password").into_response());
    }
    if password_param.password == password_param.password_confirm {
        if cfg!(debug_assertions) {
            Ok(Redirect::to("/success").into_response())
        } else {
            let args = &["register_password.sh"];
            match common::exec_command_stdin(args, &password_param.password).await {
                Ok(o) => o,
                Err(_) => return Ok(Redirect::to("/").into_response()),
            };
            Ok(Redirect::to("/success").into_response())
        }
    } else {
        match session.insert("confirm_error", true).await {
            Ok(_) => {}
            Err(_) => return Ok(Redirect::to("/").into_response()),
        }
        Ok(Redirect::to("/new_password").into_response())
    }
}

async fn success(session: Session) -> PageResult {
    session.flush().await?;
    let template = SuccessTemplate {
        version: common::VERSION,
    };
    Ok(RawTemplate(template).into_response())
}
