// SPDX-License-Identifier: MIT

use anyhow::Context;
use axum::{
    body::Bytes,
    response::IntoResponse,
    routing::{delete, post},
    Router,
};
use axum_typed_multipart::{FieldData, TryFromMultipart, TypedMultipart};

use crate::common::{self, CheckAuthRestApi, RestApiPermissionDdnsAdmin};
use crate::error::ErrorStringResult;

#[derive(TryFromMultipart)]
struct DdnsParam {
    ddns_service: String,
    hostname: Option<String>,
    username: Option<String>,
    password: Option<String>,
    proxied: Option<String>,
    interval: Option<String>,
    conf: Option<FieldData<Bytes>>,
}

pub fn routes() -> Router {
    Router::new()
        .route("/api/ddns/setup", post(rest_ddns_setup))
        .route("/api/ddns/delete", delete(rest_ddns_delete))
}

/// POST "/api/ddns/setup"
/// - Access: DdnsAdmin
/// - Input:
///   ddns_service = "no-ip", "dyndns", "cloudflare", "duckdns", or "file"
///   hostname = <target hostname>
///   username = <username> (optional for some services)
///   password = <password/API key/token>
///   proxied = <true/false> (optional, for Cloudflare)
///   interval = <ip check interval in seconds>
///   conf = <ddns config file path>
/// - Output: None
async fn rest_ddns_setup(
    _auth: CheckAuthRestApi<RestApiPermissionDdnsAdmin>,
    TypedMultipart(ddns_param): TypedMultipart<DdnsParam>,
) -> ErrorStringResult {
    let mut args = vec!["ddns_setup.sh", &ddns_param.ddns_service];

    // 基本的な引数を準備
    match &ddns_param {
        DdnsParam {
            ddns_service,
            conf: Some(conf),
            ..
        } if ddns_service == "file" => {
            // 設定ファイルアップロード処理
            let conf_path = "/etc/atmark/abos_web/inadyn.conf";
            tokio::fs::write(conf_path, &conf.contents)
                .await
                .context("Failed to write /etc/atmark/abos_web/inadyn.conf")?;
        }
        DdnsParam {
            hostname: Some(hostname),
            password: Some(password),
            interval: Some(interval),
            username,
            proxied,
            ..
        } => {
            args.extend_from_slice(&[hostname, "--password", password, "--interval", interval]);
            if let Some(user) = username {
                args.extend_from_slice(&["--user", user])
            };
            if let Some(proxied) = proxied {
                args.extend_from_slice(&["--proxied", proxied])
            };
        }
        _ => {
            Err("Invalid argument")?;
        }
    }

    // Execute setup command
    common::exec_command(&["ddns_delete.sh"]).await?;
    common::exec_command(&args).await?;
    Ok(().into_response())
}

/// DELETE "/api/ddns/delete"
/// - Access: DdnsAdmin
/// - Input: None
/// - Output: None
async fn rest_ddns_delete(
    _auth: CheckAuthRestApi<RestApiPermissionDdnsAdmin>,
) -> ErrorStringResult {
    let args = &["ddns_delete.sh"];
    common::exec_command(args).await?;

    Ok(().into_response())
}
