// SPDX-License-Identifier: MIT

use axum::{
    extract::{Path, Query},
    response::IntoResponse,
    routing::{get, post},
    Router,
};
use serde::Deserialize;

use crate::common::{
    self, container::ContainerInfo, json_response, json_stream_command, CheckAuthRestApi,
    CommandOpts, RestApiPermissionContainerAdmin, RestApiPermissionContainerView,
};
use crate::error::ErrorStringResult;

pub fn routes() -> Router {
    Router::new()
        .route("/api/containers", get(rest_container_list))
        .route(
            "/api/containers/{container}/start",
            post(rest_container_start),
        )
        .route(
            "/api/containers/{container}/stop",
            post(rest_container_stop),
        )
        .route("/api/containers/{container}/logs", get(rest_container_logs))
}

/// GET "/api/containers"
/// - Access: container view
/// - Input: None
/// - Output: ContainerList == json object with 'containers', an array of objects
///   with {id, name, state, command, image}
async fn rest_container_list(
    _auth: CheckAuthRestApi<RestApiPermissionContainerView>,
) -> ErrorStringResult {
    let container_info = ContainerInfo::get().await?;
    json_response(&container_info)
}

/// POST "/api/containers/{container}/start" (restarts container)
/// - Access: container admin
/// - Input: None
/// - Output: None
async fn rest_container_start(
    _auth: CheckAuthRestApi<RestApiPermissionContainerAdmin>,
    Path(container): Path<String>,
) -> ErrorStringResult {
    let args = &["podman.sh", "start", &container];
    common::exec_command(args).await?;

    Ok(().into_response())
}

/// POST "/api/containers/{container}/stop
/// - Access: container admin
/// - Input: None
/// - Output: None
async fn rest_container_stop(
    _auth: CheckAuthRestApi<RestApiPermissionContainerAdmin>,
    Path(container): Path<String>,
) -> ErrorStringResult {
    let args = &["podman.sh", "stop", &container];
    common::exec_command(args).await?;

    Ok(().into_response())
}

#[derive(Deserialize)]
#[serde(deny_unknown_fields)]
struct ContainerLogsFollow {
    follow: Option<bool>,
}

/// GET "/api/containers/{container}/logs
/// - Access: container view
/// - Input: optional 'follow=true' input
/// - Output: raw data
async fn rest_container_logs(
    _auth: CheckAuthRestApi<RestApiPermissionContainerView>,
    Path(container): Path<String>,
    Query(param): Query<ContainerLogsFollow>,
) -> impl IntoResponse {
    let mut args = vec!["podman.sh".to_string(), "logs".to_string(), container];
    if param.follow.unwrap_or(false) {
        args[1] = "logs_follow".to_string();
    }
    json_stream_command(args, &CommandOpts::default(), None)
}
