// SPDX-License-Identifier: MIT

use askama::Template;
use axum::{
    extract::{Extension, Form},
    middleware,
    response::{IntoResponse, Redirect},
    routing::{get, post},
    Router,
};
use serde::Deserialize;

use crate::common::{self, check_auth, container::ContainerInfo, get_title, HtmlTemplate, Title};
use crate::error::PageResult;

#[cfg(feature = "restapi")]
mod restapi;
#[cfg(not(feature = "restapi"))]
mod restapi {
    pub fn routes() -> axum::Router {
        axum::Router::new()
    }
}

#[derive(Deserialize)]
struct ContainerParam {
    name: String,
}

#[derive(Template)]
#[template(path = "../src/container/templates/container.html")]
struct ContainerTemplate {
    container_info: ContainerInfo,
}

#[derive(Template)]
#[template(path = "../src/container/templates/container_logs.html")]
struct ContainerLogsTemplate {
    container_name: String,
    container_logs: String,
}

pub fn routes() -> Router {
    Router::new()
        .route("/container", get(container))
        .route("/container_start", post(container_start))
        .route("/container_stop", post(container_stop))
        .route("/container_log", get(container_log))
        .route_layer(middleware::from_fn(check_auth))
        .merge(restapi::routes())
        .route_layer(middleware::from_fn(|request, next| {
            get_title(request, next, "./container")
        }))
}

async fn container(Extension(title): Extension<Title>) -> PageResult {
    let container_info = ContainerInfo::get().await?;
    let template = ContainerTemplate { container_info };
    Ok(HtmlTemplate::new(title.0, template).into_response())
}

async fn container_start(Form(container_param): Form<ContainerParam>) -> PageResult {
    let args = &["podman.sh", "start", &container_param.name];
    common::exec_command(args).await?;

    Ok(Redirect::to("/container").into_response())
}

async fn container_stop(Form(container_param): Form<ContainerParam>) -> PageResult {
    let args = &["podman.sh", "stop", &container_param.name];
    common::exec_command(args).await?;

    Ok(Redirect::to("/container").into_response())
}

async fn container_log(Form(container_param): Form<ContainerParam>) -> PageResult {
    let args = &["podman.sh", "logs_merged", &container_param.name];
    let output = common::exec_command(args).await?;

    let template = ContainerLogsTemplate {
        container_name: container_param.name,
        container_logs: String::from_utf8_lossy(&output.stdout).trim().to_string(),
    };
    Ok(HtmlTemplate::new("コンテナログ", template).into_response())
}
