// SPDX-License-Identifier: MIT

use crate::common::exec_command;
use anyhow::Result;
use serde::Serialize;

#[derive(Debug, Serialize)]
pub struct UsbDeviceInfo {
    pub devices: Vec<UsbDevice>,
}

#[derive(Debug, PartialEq, Eq, Serialize)]
pub struct UsbDevice {
    pub list_id: String,
    pub permission: String,
    pub bus_device_num: String,
    pub vendor_id: String,
    pub product_id: String,
    pub model: String,
    pub usb_interfaces: String,
    pub serial: String,
    pub dir_pass: String,
}

impl UsbDeviceInfo {
    pub fn from_list_devices(output: &str) -> Result<Self> {
        let mut devices = Vec::new();
        if output != "No USB devices connected\n" {
            for device in output.split('\n') {
                if device.is_empty() {
                    continue;
                }
                let props = device.split_whitespace().collect::<Vec<&str>>();
                let default_str = &"--";
                let list_id = props.first().unwrap_or(default_str).to_string();
                let permission = props.get(1).unwrap_or(default_str).to_string();
                let bus_device_num = props.get(2).unwrap_or(default_str).to_string();
                let vendor_id = props.get(3).unwrap_or(default_str).to_string();
                let product_id = props.get(4).unwrap_or(default_str).to_string();
                let model = props.get(5).unwrap_or(default_str).to_string();
                let usb_interfaces = props.get(6).unwrap_or(default_str).to_string();
                let serial = props.get(7).unwrap_or(default_str).to_string();
                let dir_pass = props.get(8).unwrap_or(default_str).to_string();
                devices.push(UsbDevice {
                    list_id,
                    permission,
                    bus_device_num,
                    vendor_id,
                    product_id,
                    model,
                    usb_interfaces,
                    serial,
                    dir_pass,
                });
            }
            devices.sort_unstable_by(|a, b| a.list_id.cmp(&b.list_id));
        }
        Ok(UsbDeviceInfo { devices })
    }

    pub async fn get() -> Result<Self> {
        let args = &["usb_filter.sh", "list_devices"];
        UsbDeviceInfo::from_list_devices(&String::from_utf8_lossy(
            &exec_command(args).await?.stdout,
        ))
    }
}

#[derive(Debug, Serialize)]
pub struct AllowRulesInfo {
    pub devices: Vec<AllowRulesDevice>,
    pub classes: Vec<AllowRulesClass>,
    pub all_class_allowed: bool,
}

#[derive(Debug, PartialEq, Eq, Serialize)]
pub struct AllowRulesDevice {
    pub list_id: String,
    pub vendor_id: String,
    pub product_id: String,
    pub model: String,
    pub usb_interfaces: String,
    pub serial: String,
}

#[derive(Debug, PartialEq, Eq, Serialize)]
pub struct AllowRulesClass {
    pub list_id: String,
    pub class: String,
}

impl AllowRulesInfo {
    pub fn from_list_rules(rules: &str, block_classes: &str) -> Result<Self> {
        let mut allow_rules_devices = Vec::new();
        let mut allow_rules_classes = Vec::new();
        for rule in rules.split('\n') {
            if rule.is_empty() {
                continue;
            }
            let props = rule
                .split_whitespace()
                .map(|x| x.replace("\"", ""))
                .collect::<Vec<String>>();
            let default_str = &"--".to_string();
            match props.get(1).map(|x| x.as_str()) {
                Some("device") => {
                    let list_id = props.first().unwrap_or(default_str).to_string();
                    let vendor_id = props.get(2).unwrap_or(default_str).to_string();
                    let product_id = props.get(3).unwrap_or(default_str).to_string();
                    let model = props.get(4).unwrap_or(default_str).to_string();
                    let usb_interfaces = props.get(5).unwrap_or(default_str).to_string();
                    let serial = props.get(6).unwrap_or(default_str).to_string();
                    allow_rules_devices.push(AllowRulesDevice {
                        list_id,
                        vendor_id,
                        product_id,
                        model,
                        usb_interfaces,
                        serial,
                    });
                }
                Some("class") => {
                    let list_id = props.first().unwrap_or(default_str).to_string();
                    let class = props.get(2).unwrap_or(default_str).to_string();
                    allow_rules_classes.push(AllowRulesClass { list_id, class })
                }
                _ => continue, /* comment or new rule type we're not handling */
            }
        }
        allow_rules_devices.sort_unstable_by(|a, b| a.list_id.cmp(&b.list_id));
        allow_rules_classes.sort_unstable_by(|a, b| a.list_id.cmp(&b.list_id));
        let mut all_class_allowed = false;
        if block_classes.is_empty() {
            all_class_allowed = true;
        }
        Ok(AllowRulesInfo {
            devices: allow_rules_devices,
            classes: allow_rules_classes,
            all_class_allowed,
        })
    }

    pub async fn get() -> Result<Self> {
        let list_rules_args = &["usb_filter.sh", "list_rules"];
        let list_block_classes_args = &["usb_filter.sh", "list_block_classes"];
        AllowRulesInfo::from_list_rules(
            &String::from_utf8_lossy(&exec_command(list_rules_args).await?.stdout),
            &String::from_utf8_lossy(&exec_command(list_block_classes_args).await?.stdout),
        )
    }
}
