// SPDX-License-Identifier: MIT

use anyhow::Result;
use serde::{Deserialize, Serialize};

use crate::common::{RestApiPermission, RestApiToken};

#[derive(Serialize)]
pub struct RestApiSerializeToken<'a> {
    #[serde(skip_serializing_if = "Option::is_none")]
    token: Option<&'a str>,
    permissions: &'a [RestApiPermission],
}

pub struct SerializeTokenOptions {
    pub include_token_id: bool,
}

impl RestApiToken {
    pub fn to_serialize<'a>(
        &'a self,
        options: &SerializeTokenOptions,
    ) -> RestApiSerializeToken<'a> {
        RestApiSerializeToken {
            token: options.include_token_id.then_some(&self.token_id),
            permissions: &self.permissions,
        }
    }
}

pub fn serialize_token(token: &RestApiToken, options: &SerializeTokenOptions) -> Result<String> {
    Ok(serde_json::to_string(&token.to_serialize(options))?)
}

#[derive(Deserialize)]
pub struct RestApiDeserializeToken {
    pub permissions: Option<Vec<RestApiPermission>>,
}

impl RestApiDeserializeToken {
    pub fn into_token(self, token_id: String) -> RestApiToken {
        RestApiToken {
            token_id,
            permissions: self
                .permissions
                .unwrap_or_else(|| vec![RestApiPermission::Admin]),
        }
    }
}

pub fn parse_token(file_name: String, json: &str) -> Result<RestApiToken> {
    let token: RestApiDeserializeToken = serde_json::from_str(json)?;
    Ok(token.into_token(file_name))
}
