// SPDX-License-Identifier: MIT

use paste::paste;
use serde::{Deserialize, Serialize};
use std::fmt;

use crate::common::Config;

// get permission from type data, see CheckAuthRestApi comment
pub trait ToRestApiPermission {
    fn to_permission() -> RestApiPermission;
}

// Because we use permissions in type templates we define permissions twice through a macro:
// - RestApiPermission is an enum that containers all permissions e.g.
//   RestApiPermission::Admin
// - an extra RestApiPermissionXYZ type (e.g. RestApiPermissionAdmin) to be used
//   with CheckAuthRestApi
//
// The macro is a way of not repeating ourselves too much.
// We need paste! to creates tokens for RestApiPermissionXYZ ( [< and >] blocks)
macro_rules!define_permissions {
    ($(($perm:ident, $titles:expr)),+) => {
        paste! {
            #[derive(Clone, Copy, Debug, PartialEq, Eq, Hash, Deserialize, Serialize)]
            pub enum RestApiPermission {
                $( $perm, )*
            }
            impl RestApiPermission {
                fn all_with_titles() -> &'static [(RestApiPermission, &'static [&'static str])] {
                    // No easy way to get the static slice size:
                    // https://github.com/rust-lang/lang-team/issues/28
                    const SIZE: usize = 0usize $( + { stringify!($perm); 1 } )*;
                    static PERMISSIONS: [(RestApiPermission, &'static [&'static str]); SIZE] = [
                        $( ( RestApiPermission::$perm, &$titles ), )*
                    ];
                    &PERMISSIONS
                }
            }
            $(
                pub enum [<RestApiPermission $perm>] {}

                impl ToRestApiPermission for [<RestApiPermission $perm>] {
                    fn to_permission() -> RestApiPermission {
                        RestApiPermission::$perm
                    }
                }
            )*
        }
    }
}

define_permissions!(
    (AbosWebRestart, ["./settings"]),
    (Admin, []),
    (ContainerAdmin, ["./container"]),
    (ContainerView, ["./container"]),
    (Custom, []),
    (DdnsAdmin, ["./ddns"]),
    (
        NetworkAdmin,
        ["./connection", "./dhcp", "./nat", "./wlan", "./wwan"]
    ),
    (
        NetworkView,
        ["./connection", "./dhcp", "./nat", "./wlan", "./wwan"]
    ),
    (Poweroff, ["./power"]),
    (Reboot, ["./power"]),
    (ResetDefault, ["./settings"]),
    (Sleep, ["./power"]),
    (SmsView, ["./wwan"]),
    (SmsSend, ["./wwan"]),
    (SwuInstall, ["./swu"]),
    (SwuView, ["./swu"]),
    (TimeAdmin, ["./time"]),
    (TimeView, ["./time"]),
    (VpnAdmin, ["./vpn"])
);

impl fmt::Display for RestApiPermission {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        fmt::Debug::fmt(self, f)
    }
}

impl RestApiPermission {
    pub fn all() -> Vec<RestApiPermission> {
        RestApiPermission::all_with_titles()
            .iter()
            .filter_map(|(perm, titles)| {
                if titles.is_empty()
                    || titles.iter().any(|&title| {
                        Config::get()
                            .customize
                            .get()
                            .get_page_title(title)
                            .is_some()
                    })
                {
                    Some(*perm)
                } else {
                    None
                }
            })
            .collect()
    }
}
