// SPDX-License-Identifier: MIT

use anyhow::{Error, Result};
use std::path::PathBuf;

mod check;
mod fs;
mod json;
mod json_stream;
pub mod permissions;
mod serialize;

pub use check::CheckAuthRestApi;
pub use json::{json_response, json_response_raw, JsonOrForm, JsonOrFormOption};
pub use json_stream::{json_stream_command, multipart_to_input_chan};
use permissions::RestApiPermission;
pub use serialize::{parse_token, serialize_token, RestApiSerializeToken, SerializeTokenOptions};

#[cfg(debug_assertions)]
const TOKENS_PATH: &str = "tokens";
#[cfg(not(debug_assertions))]
const TOKENS_PATH: &str = "/var/lib/abos-web/tokens";

#[derive(Clone, Debug, PartialEq)]
pub struct RestApiToken {
    pub token_id: String,
    pub permissions: Vec<RestApiPermission>,
    // eventually add source IP checks or other limits
}

pub async fn check_token(token_id: &str, permission: RestApiPermission) -> Result<()> {
    let token = get_token(token_id).await?;
    if !token
        .permissions
        .iter()
        .any(|&p| p == RestApiPermission::Admin || p == permission)
    {
        return Err(Error::msg("Not enough permission"));
    }
    Ok(())
}

pub async fn get_token(token_id: &str) -> Result<RestApiToken> {
    fs::get_token(&PathBuf::from(TOKENS_PATH), token_id).await
}

pub async fn list_tokens() -> Result<Vec<RestApiToken>> {
    fs::list_tokens(&PathBuf::from(TOKENS_PATH)).await
}

pub async fn write_token(token: &RestApiToken) -> Result<()> {
    fs::write_token(&PathBuf::from(TOKENS_PATH), token).await
}

pub async fn create_token(permissions: Vec<RestApiPermission>) -> Result<RestApiToken> {
    fs::create_token(&PathBuf::from(TOKENS_PATH), permissions).await
}

pub async fn delete_token(token_id: &str) -> Result<()> {
    fs::delete_token(&PathBuf::from(TOKENS_PATH), token_id).await
}
