// SPDX-License-Identifier: MIT

use anyhow::{Context, Error, Result};
use async_stream::try_stream;
use axum::{
    body::Body,
    extract::Multipart,
    http::header::{HeaderName, CONTENT_TYPE},
};
use futures::Stream;
use serde::Serialize;
use std::os::unix::process::ExitStatusExt;
use tokio::sync::mpsc;
use tracing::{debug, warn};

use crate::common::process::{
    stream_command_input, CommandOpts, InputStream, OutputChannel, OutputSink,
};

// structs for serialization
#[derive(Serialize)]
struct StdoutLine<'a> {
    stdout: &'a str,
}
#[derive(Serialize)]
struct StderrLine<'a> {
    stderr: &'a str,
}
#[derive(Serialize)]
struct ErrorLine<'a> {
    error: &'a str,
}
#[derive(Serialize)]
struct ExitCode {
    #[serde(skip_serializing_if = "Option::is_none")]
    exit_code: Option<i32>,
    #[serde(skip_serializing_if = "Option::is_none")]
    exit_signal: Option<i32>,
}

/// InputStream implementation for rest
impl InputStream for &mut mpsc::Receiver<Vec<u8>> {
    async fn next_chunk(&mut self) -> Option<Vec<u8>> {
        self.recv().await
    }
}
/// OutputSink implementation for rest
impl OutputSink for &mut mpsc::Sender<Vec<u8>> {
    async fn send_line(&mut self, chan: OutputChannel, data: String) -> Result<()> {
        let serialized = match chan {
            OutputChannel::Stdout => serde_json::to_string(&StdoutLine { stdout: &data }),
            OutputChannel::Stderr => serde_json::to_string(&StderrLine { stderr: &data }),
        };
        let Ok(json) = serialized else {
            warn!("Could not serialize {}", data);
            // assume send might still work
            return Ok(());
        };
        self.send(json.into_bytes())
            .await
            .context("Receiver closed?")
    }
    async fn check_output(&mut self) -> Result<()> {
        if self.is_closed() {
            return Err(Error::msg("Client closed"));
        }
        Ok(())
    }
}

/// return a stream suitable for Body::from_stream from a mpsc receiver
fn forward_jsonify_output(mut rx: mpsc::Receiver<Vec<u8>>) -> impl Stream<Item = Result<Vec<u8>>> {
    try_stream! {
        while let Some(line) = rx.recv().await {
            yield line;
            // separate json by new lines
            yield vec![b'\n'];
        }
    }
}

pub async fn multipart_to_input_chan(mut input: Multipart) -> mpsc::Receiver<Vec<u8>> {
    let (tx, rx) = mpsc::channel(20);
    tokio::spawn(async move {
        // input_stream borrows a ref on input, and we apparently cannot move both input_stream and
        // input together in the tokio closure, so we cannot check if we could find a field before
        // actually spawning this task...
        // If the user didn't send anything then swupdate will be spawned from stdin but not get
        // any data and fail immediately, so it's acceptable enough.
        if let Ok(Some(mut input_stream)) = input.next_field().await {
            loop {
                match input_stream.chunk().await {
                    Ok(Some(chunk)) => {
                        if let Err(e) = tx.send(chunk.into()).await {
                            debug!("Failed forwarding input to command: {:?}", e);
                            break;
                        }
                    }
                    Ok(None) => break, // end of stream
                    Err(e) => {
                        debug!(
                            "Stream interrupted unexpectedly when forwarding to command: {:?}",
                            e
                        );
                        break;
                    }
                }
            }
        }
    });
    rx
}

/// Run given command and returns a reponse suitable for REST api streaming
/// Since this runs asynchronously it must have full ownership of args, and thus
/// does not allow &str like other exec functions
pub fn json_stream_command(
    args: Vec<String>,
    opts: &CommandOpts,
    mut input: Option<mpsc::Receiver<Vec<u8>>>,
) -> ([(HeaderName, &'static str); 1], Body) {
    let chan = {
        let (mut tx, rx) = mpsc::channel(20);
        let local_opts = opts.clone();
        tokio::spawn(async move {
            match stream_command_input(&args, &local_opts, input.as_mut(), &mut tx).await {
                Err(e) => {
                    let errmsg = format!("{e:#}");
                    if let Ok(json) = serde_json::to_string(&ErrorLine { error: &errmsg }) {
                        let _ = tx.send(json.into_bytes()).await;
                    }
                }
                Ok(status) => {
                    let code = ExitCode {
                        exit_code: status.code(),
                        exit_signal: status.signal(),
                    };
                    if let Ok(json) = serde_json::to_string(&code) {
                        let _ = tx.send(json.into_bytes()).await;
                    }
                }
            }
        });
        rx
    };
    (
        [(CONTENT_TYPE, "application/json")],
        Body::from_stream(forward_jsonify_output(chan)),
    )
}
