// SPDX-License-Identifier: MIT

use anyhow::{anyhow, Context, Error, Result};
use axum::extract::ws::WebSocket;
use futures::stream::SplitStream;
use std::io::Cursor;
use std::path::Path;
use std::process::{ExitStatus, Output, Stdio};
use tokio::io::{AsyncBufReadExt, AsyncWriteExt, BufReader};
use tokio::process::{ChildStdin, Command};
use tokio::time;
use tracing::{info, trace};

#[cfg(debug_assertions)]
const SCRIPTS_DIR: &str = "./scripts";

#[cfg(not(debug_assertions))]
const SCRIPTS_DIR: &str = "/usr/libexec/abos-web";

fn doas() -> &'static str {
    if cfg!(debug_assertions) && Path::new("./utils/ssh_run.sh").is_file() {
        "./utils/ssh_run.sh"
    } else if Path::new("/usr/bin/doas").is_file() {
        "doas"
    } else {
        ""
    }
}

// take script_dir / doas as argument for tests
fn _command<S>(args: &[S], script_dir: Option<&str>, doas: &str) -> Result<Command>
where
    S: AsRef<std::ffi::OsStr>,
{
    if args.is_empty() {
        return Err(anyhow!("no arg"));
    }

    let mut alloc = String::new();
    let arg0 = match script_dir {
        None => args[0].as_ref(),
        Some(dir) => {
            alloc.push_str(dir);
            alloc.push('/');
            alloc.push_str(
                args[0]
                    .as_ref()
                    .to_str()
                    .context("Could not cast OsStr back to str")?,
            );
            alloc.as_ref()
        }
    };

    if doas.is_empty() {
        let mut command = Command::new(arg0);
        command.args(&args[1..]);
        Ok(command)
    } else {
        let mut command = Command::new(doas);
        command.arg(arg0);
        command.args(&args[1..]);
        Ok(command)
    }
}

#[derive(Clone)]
pub struct CommandOpts {
    /// Is the command a script in abos-web libexec dir?
    pub is_script: bool,
    /// Should the command run as root (with doas)?
    pub as_root: bool,
    /// For streams, should the streaming stop when the client disconnects?
    pub stream_ignore_output_errors: bool,
}

impl Default for CommandOpts {
    fn default() -> Self {
        CommandOpts {
            is_script: true,
            as_root: true,
            stream_ignore_output_errors: false,
        }
    }
}

pub fn command<S>(args: &[S], opts: &CommandOpts) -> Result<Command>
where
    S: AsRef<std::ffi::OsStr>,
{
    let scripts_dir = if opts.is_script {
        Some(SCRIPTS_DIR)
    } else {
        None
    };
    let doas = if opts.as_root { doas() } else { "" };

    _command(args, scripts_dir, doas)
}

/// Wrapped input for stream_command_input:
/// this is useful to reuse stream_command with web socket and rest API
pub trait InputStream {
    async fn next_chunk(&mut self) -> Option<Vec<u8>>;
}
/// Specify whether log was sent to stdout or stderr
pub enum OutputChannel {
    Stdout,
    Stderr,
}
/// Wrapped output for stream_command*
/// this is useful to reuse stream_command with web socket and rest API
pub trait OutputSink {
    /// send line to client
    async fn send_line(&mut self, chan: OutputChannel, data: String) -> Result<()>;
    /// checks output is still valid - this is used to detect closed outputs even
    /// if there is nothing to send (mut for WS, which has no shared ref check)
    async fn check_output(&mut self) -> Result<()>;
}

/// get next chunk from input if input is set.
/// This is necessary to allow having an optional input in select!,
/// even if we never call it with a none input.
async fn next_if_some<S>(input: &mut Option<S>) -> Option<Vec<u8>>
where
    S: InputStream,
{
    match input {
        None => None,
        Some(ref mut input) => input.next_chunk().await,
    }
}

/// likewise: write optional data to optional process stdin.
/// Returns true if we continue or false on error
async fn write_if_some(stdin: &mut Option<ChildStdin>, data: &mut Option<Cursor<Vec<u8>>>) -> bool {
    let Some(stdin) = stdin.as_mut() else {
        info!("stream command helper write_if_some called with None input?");
        return false;
    };
    if let Err(e) = stdin.write_all_buf(&mut data.as_mut().unwrap()).await {
        info!("Failed to write chunk to process {:?}", e);
        return false;
    };
    true
}

/// Run given command and feed  stdout/stderr to output.
/// If input is not none, feed it to command.
/// Command runs to completion even if output closes early,
/// and this returns exit code.
pub async fn stream_command_input<S, I, O>(
    args: &[S],
    opts: &CommandOpts,
    mut input: Option<I>,
    mut output: O,
) -> Result<ExitStatus>
where
    S: AsRef<std::ffi::OsStr> + std::fmt::Debug,
    I: InputStream,
    O: OutputSink,
{
    trace!("Executing (stream) {:?}", args);
    let mut command = command(args, opts)?
        .stdin(if input.is_some() {
            Stdio::piped()
        } else {
            Stdio::null()
        })
        .stdout(Stdio::piped())
        .stderr(Stdio::piped())
        .spawn()
        .with_context(|| format!("Could not start command: {args:?}"))?;

    let mut stdin = if input.is_some() {
        Some(command.stdin.take().context("command has no stdin")?)
    } else {
        None
    };
    let mut stdin_data = None;

    let mut stdout = BufReader::new(command.stdout.take().context("command has no stdout")?);
    let mut stdout_buf = vec![];
    let mut stdout_open = true;

    let mut stderr = BufReader::new(command.stderr.take().context("command has no stderr")?);
    let mut stderr_buf = vec![];
    let mut stderr_open = true;

    while stdout_open || stderr_open {
        tokio::select! {
            n = stdout.read_until(b'\n', &mut stdout_buf), if stdout_open => {
                if n.unwrap_or(0) == 0 && stdout_buf.is_empty() {
                    stdout_open = false;
                } else {
                    let text = String::from_utf8_lossy(&stdout_buf);
                    trace!("{}", text);
                    if let Err(e) = output.send_line(OutputChannel::Stdout, text.into_owned()).await {
                        if ! opts.stream_ignore_output_errors {
                            return Err(e)
                        }
                    }
                    stdout_buf = vec![];
                }
            },
            n = stderr.read_until(b'\n', &mut stderr_buf), if stderr_open => {
                if n.unwrap_or(0) == 0 && stderr_buf.is_empty() {
                    stderr_open = false;
                } else {
                    let text = String::from_utf8_lossy(&stderr_buf);
                    trace!("{}", text);
                    if let Err(e) = output.send_line(OutputChannel::Stderr, text.into_owned()).await {
                        if ! opts.stream_ignore_output_errors {
                            return Err(e)
                        }
                    }
                    stderr_buf = vec![];
                }
            },
            _ = time::sleep(time::Duration::from_secs(5)), if ! opts.stream_ignore_output_errors => {
                output.check_output().await?;
            }
            chunk = next_if_some(&mut input), if stdin.is_some() && stdin_data.is_none() => {
                match chunk {
                    None => {
                        // drop stdin to close it, so process can exit
                        stdin = None;
                    }
                    Some(data) => {
                        stdin_data = Some(Cursor::new(data));
                    },
                }
            },
            cont = write_if_some(&mut stdin, &mut stdin_data), if stdin_data.is_some() => {
                if cont {
                    stdin_data = None
                } else {
                    stdin = None;
                    stdin_data = None
                }
            }
        }
    }

    command.wait().await.context("waiting failed")
}

/// Version of 'stream_command' without input: this is only here because rust
/// requires the None parameter to stream_input_command to have a specific type
/// for async code, and casting it every time is cumbersome.
pub async fn stream_command<S, O>(args: &[S], opts: &CommandOpts, output: O) -> Result<ExitStatus>
where
    S: AsRef<std::ffi::OsStr> + std::fmt::Debug,
    O: OutputSink,
{
    stream_command_input(args, opts, None::<&mut SplitStream<&mut WebSocket>>, output).await
}

pub async fn exec_command_with_opts<S>(args: &[S], opts: &CommandOpts) -> Result<Output>
where
    S: AsRef<std::ffi::OsStr> + std::fmt::Debug,
{
    trace!(
        "Executing {:?} (script: {}, root: {})",
        args,
        opts.is_script,
        opts.as_root
    );
    let output = command(args, opts)?.output().await?;
    if output.status.success() {
        Ok(output)
    } else {
        Err(Error::msg(format!(
            "Command {:?} failed with {}:\nstderr:\n{}\nstdout:\n{}",
            args,
            output.status,
            String::from_utf8_lossy(&output.stderr),
            String::from_utf8_lossy(&output.stdout)
        )))
    }
}

/// Run abos-web internal script as root and returns its output
/// or an error on failure.
pub async fn exec_command<S>(args: &[S]) -> Result<Output>
where
    S: AsRef<std::ffi::OsStr> + std::fmt::Debug,
{
    exec_command_with_opts(args, &CommandOpts::default()).await
}

/// Run abos-web internal script as non root, use command,
/// and returns its output
/// or an error on failure.
pub async fn exec_command_no_script<S>(args: &[S]) -> Result<Output>
where
    S: AsRef<std::ffi::OsStr> + std::fmt::Debug,
{
    exec_command_with_opts(
        args,
        &CommandOpts {
            is_script: false,
            as_root: false,
            ..Default::default()
        },
    )
    .await
}

/// Same as exec_command, except it feeds the stdin to the process.
/// Only works for small inputs where command will not be stuck writing
/// its output
pub async fn exec_command_stdin<S>(args: &[S], input: &str) -> Result<Output>
where
    S: AsRef<std::ffi::OsStr> + std::fmt::Debug,
{
    trace!("Executing {:?} (with stdin)", args);
    // We could try to fit this into stream_command but it's simpler to run directly
    let mut command = command(args, &CommandOpts::default())?
        .stdin(Stdio::piped())
        .stdout(Stdio::piped())
        .stderr(Stdio::piped())
        .spawn()
        .context("Could not start command")?;
    let mut stdin = command.stdin.take().context("command has no stdin")?;
    stdin.write_all(input.as_bytes()).await?;
    // drop to close
    drop(stdin);
    let output = command.wait_with_output().await?;
    if output.status.success() {
        Ok(output)
    } else {
        Err(Error::msg(format!(
            "Command {:?} failed with {}:\nstderr:\n{}\nstdout:\n{}",
            args,
            output.status,
            String::from_utf8_lossy(&output.stderr),
            String::from_utf8_lossy(&output.stdout)
        )))
    }
}

#[cfg(test)]
mod tests {
    use crate::common::process::_command;
    use anyhow::Result;
    use std::process::Command;

    fn check_command(cmd: &Command, arg0: &str, args: &[&str]) {
        assert_eq!(cmd.get_program(), arg0);
        assert_eq!(cmd.get_args().count(), args.len());
        cmd.get_args()
            .zip(args)
            .for_each(|(cmd_arg, check_arg)| assert_eq!(&cmd_arg.to_str().unwrap(), check_arg));
    }

    #[test]
    fn test_command() -> Result<()> {
        assert!(_command(&Vec::<String>::new(), None, "").is_err());

        // single argument
        let args = &["ls"];
        let cmd = _command(args, None, "")?;
        check_command(cmd.as_std(), "ls", &[]);
        let cmd = _command(&args.map(|s| s.to_string()), None, "")?;
        check_command(cmd.as_std(), "ls", &[]);

        let cmd = _command(args, None, "doas")?;
        check_command(cmd.as_std(), "doas", &["ls"]);
        let cmd = _command(&args.map(|s| s.to_string()), None, "doas")?;
        check_command(cmd.as_std(), "doas", &["ls"]);

        let cmd = _command(args, Some("./scripts"), "")?;
        check_command(cmd.as_std(), "./scripts/ls", &[]);
        let cmd = _command(&args.map(|s| s.to_string()), Some("./scripts"), "")?;
        check_command(cmd.as_std(), "./scripts/ls", &[]);

        let cmd = _command(args, Some("./scripts"), "doas")?;
        check_command(cmd.as_std(), "doas", &["./scripts/ls"]);
        let cmd = _command(&args.map(|s| s.to_string()), Some("./scripts"), "doas")?;
        check_command(cmd.as_std(), "doas", &["./scripts/ls"]);

        // some argument
        let args = &["ls", "-l"];
        let cmd = _command(args, None, "")?;
        check_command(cmd.as_std(), "ls", &["-l"]);
        let cmd = _command(&args.map(|s| s.to_string()), None, "")?;
        check_command(cmd.as_std(), "ls", &["-l"]);

        let cmd = _command(args, None, "doas")?;
        check_command(cmd.as_std(), "doas", &["ls", "-l"]);
        let cmd = _command(&args.map(|s| s.to_string()), None, "doas")?;
        check_command(cmd.as_std(), "doas", &["ls", "-l"]);

        let cmd = _command(args, Some("./scripts"), "")?;
        check_command(cmd.as_std(), "./scripts/ls", &["-l"]);
        let cmd = _command(&args.map(|s| s.to_string()), Some("./scripts"), "")?;
        check_command(cmd.as_std(), "./scripts/ls", &["-l"]);

        let cmd = _command(args, Some("./scripts"), "doas")?;
        check_command(cmd.as_std(), "doas", &["./scripts/ls", "-l"]);
        let cmd = _command(&args.map(|s| s.to_string()), Some("./scripts"), "doas")?;
        check_command(cmd.as_std(), "doas", &["./scripts/ls", "-l"]);

        Ok(())
    }
}
