// SPDX-License-Identifier: MIT

use anyhow::{Error, Result};
use once_cell::sync::OnceCell;
use pwhash::sha512_crypt;
use std::time::Duration;
use tokio::sync::{Mutex, MutexGuard};
use tokio::time::Instant;

use crate::common::{self};

static PASSWORD_THROTTLE: OnceCell<Mutex<()>> = OnceCell::new();

pub fn password_throttle_init() {
    PASSWORD_THROTTLE
        .set(Mutex::new(()))
        .expect("init should only be called once");
}

pub async fn password_throttle_lock() -> MutexGuard<'static, ()> {
    PASSWORD_THROTTLE
        .get()
        .expect("password throttle should be initialized")
        .lock()
        .await
}

pub async fn current_password_hash() -> Result<String> {
    if cfg!(debug_assertions) {
        return Ok("!".to_string());
    };
    let args = vec!["get_pw_hash.sh"];
    let output = common::exec_command(&args).await?;
    Ok(String::from_utf8_lossy(&output.stdout).replace('\n', ""))
}

pub async fn check_password(password: &str) -> Result<()> {
    let _lock = password_throttle_lock().await;
    let start_time = Instant::now();
    let hash = current_password_hash().await?;
    // always allow login if password is unset
    if hash != "!" && !sha512_crypt::verify(password, &hash) {
        tokio::time::sleep_until(start_time + Duration::from_secs(2)).await;
        Err(Error::msg("Invalid password"))?
    }
    Ok(())
}
