// SPDX-License-Identifier: MIT

use anyhow::{Context, Error, Result};
use serde::Serialize;
use std::collections::{BTreeMap, HashMap};

use crate::common::exec_command;

/// a NetworkManager connection.
/// Key settings (obtainable immediately through `nmcli c`) are listed immediately,
/// other settings are only valid if props is some.
#[derive(Debug, PartialEq, Serialize)]
pub struct NMCon {
    pub name: String,
    // XXX make state an enum?
    // we don't really care about active connections so keep string for now
    pub state: String,
    pub uuid: String,
    pub ctype: String,
    #[serde(skip_serializing_if = "Option::is_none")]
    pub device: Option<String>,
    #[serde(skip_serializing_if = "Option::is_none")]
    pub props: Option<BTreeMap<String, String>>,
}

impl NMCon {
    /// parse one line of 'nmcli -t -c no -f name,uuid,type,device,state con'
    fn from_con_list_line(output: &str) -> Result<Self> {
        let fields: Vec<String> = parse_nmcli_t_line(output);
        let Ok([name, uuid, ctype, device, state]) = <[String; 5]>::try_from(fields) else {
            return Err(Error::msg(format!("line did not have 5 fields: {output}")));
        };
        Ok(NMCon {
            name,
            uuid,
            ctype,
            device: if device.is_empty() {
                None
            } else {
                Some(device)
            },
            state,
            props: None::<BTreeMap<String, String>>,
        })
    }
    /// parse output of `nmcli -t -c no c show <con>`
    fn from_con_show(output: &str) -> Result<Self> {
        let props = output
            .trim()
            .split('\n')
            .map(|line| {
                line.split_once(':')
                    .map(|(a, b)| (a.to_string(), b.to_string()))
                    .with_context(|| format!("Line did not contain ':': {line}"))
            })
            .collect::<Result<BTreeMap<String, String>>>()?;
        Ok(NMCon {
            name: props
                .get("connection.id")
                .or_else(|| props.get("GENERAL.CONNECTION"))
                .context("No connection.id or GENERAL.CONNECTION in device")?
                .clone(),
            // CON-PATH can be used as uuid
            uuid: props
                .get("connection.uuid")
                .or_else(|| props.get("GENERAL.CON-PATH"))
                .context("No connection.uuid or GENERAL.CON-PATH in device")?
                .clone(),
            ctype: props
                .get("connection.type")
                .or_else(|| props.get("GENERAL.TYPE"))
                .context("No connection.type or GENERAL.TYPE in device")?
                .clone(),
            device: props
                .get("connection.interface-name")
                .or_else(|| props.get("GENERAL.DEVICE"))
                .map(|i| i.to_string()),
            state: props.get("GENERAL.STATE").cloned().unwrap_or_default(),
            props: Some(props),
        })
    }

    pub async fn from_id(id: &str) -> Result<Self> {
        let args = &["nmcli_show.sh", "show", id];
        NMCon::from_con_show(&String::from_utf8_lossy(&exec_command(args).await?.stdout))
    }

    pub async fn from_device(dev: &str) -> Result<Self> {
        let args = &["nmcli_show.sh", "show_dev", dev];
        NMCon::from_con_show(&String::from_utf8_lossy(&exec_command(args).await?.stdout))
    }

    /// get prop or return an error
    /// (helper to use ?)
    pub fn prop(&self, key: &str) -> Option<&str> {
        self.props
            .as_ref()
            .and_then(|props| props.get(key).map(|v| v.as_str()))
    }

    /// get prop as owned value (or error)
    pub fn prop_take(&mut self, key: &str) -> Option<String> {
        self.props.as_mut().and_then(|props| props.remove(key))
    }
}

pub fn parse_nmcli_t_line(source: &str) -> Vec<String> {
    // Should use a real parser like 'nom', hand-coded solution
    // in two passes for now:
    // - split on colons except if it was prefixed by a backslash
    // - remove backslashes
    let mut split_was_backslash = false;
    let mut filter_was_backslash = false;
    let fields: Vec<String> = source
        .trim()
        .split(|c: char| {
            if split_was_backslash {
                split_was_backslash = false;
                false
            } else if c == '\\' {
                split_was_backslash = true;
                false
            } else {
                c == ':'
            }
        })
        .map(|s| {
            s.chars()
                .filter(|c: &char| {
                    if filter_was_backslash {
                        filter_was_backslash = false;
                        true
                    } else if *c == '\\' {
                        filter_was_backslash = true;
                        false
                    } else {
                        true
                    }
                })
                .collect()
        })
        .collect();

    fields
}

/// parse 'nmcli -t -c no -f name,uuid,type,device,state con'
/// (can't define an impl for an external type)
fn nmconvec_from_con_list(output: &str) -> Result<Vec<NMCon>> {
    // note this (collect of Result) fails on first error:
    // should we ignore errors instead?
    output
        .trim()
        .split('\n')
        .map(NMCon::from_con_list_line)
        .collect()
}

pub async fn nmconvec() -> Result<Vec<NMCon>> {
    let args = &["nmcli_show.sh", "list"];
    nmconvec_from_con_list(&String::from_utf8_lossy(&exec_command(args).await?.stdout))
}

pub async fn nmcon_mod(id: String, props: HashMap<String, String>) -> Result<()> {
    // need to pre-allocate strings to use as refs in args vec
    // (HashMap iter() only gives us &String, which would otherwise require cloning)
    let mut args = vec!["nmcli.sh", "modify", &id];
    args.append(
        &mut props
            .iter()
            .flat_map(|(k, v)| [k.as_str(), v.as_str()])
            .collect::<Vec<&str>>(),
    );
    let _ = exec_command(&args).await?;
    Ok(())
}

pub async fn nmcon_act(action: &str, id: &str) -> Result<()> {
    let script = if ["show", "show_dev"].contains(&action) {
        "nmcli_show.sh"
    } else {
        "nmcli.sh"
    };
    let args = &[script, action, id];
    let _ = exec_command(args).await?;
    Ok(())
}

#[cfg(test)]
mod tests {
    use super::*;
    use anyhow::Result;
    struct LineData {
        line: &'static str,
        result: Vec<&'static str>,
    }
    #[test]
    fn test_parse_line() {
        let test_data = vec![
            LineData {
                line: "Wired connection 2:8fcccf6d-776c-3bd7-a7f9-4433dbcc4361:802-3-ethernet::",
                result: vec![
                    "Wired connection 2",
                    "8fcccf6d-776c-3bd7-a7f9-4433dbcc4361",
                    "802-3-ethernet",
                    "",
                    "",
                ],
            },
            LineData {
                line: r#"test AP:83:36:130 Mbit/s:WPA2:36\:6F\:24\:C0\:C7\:F3"#,
                result: vec![
                    "test AP",
                    "83",
                    "36",
                    "130 Mbit/s",
                    "WPA2",
                    "36:6F:24:C0:C7:F3",
                ],
            },
        ];
        for test in test_data {
            let result = parse_nmcli_t_line(test.line);
            assert_eq!(result, test.result);
        }
    }
    #[test]
    fn test_parse_list() -> Result<()> {
        // single con with device set
        let con1 = NMCon::from_con_list_line(
            "Wired connection 1:18d241f1-946c-3325-974f-65cda3e6eea5:802-3-ethernet:eth0:active",
        )?;
        assert_eq!(con1.name, "Wired connection 1");
        assert_eq!(con1.device.as_ref().unwrap(), "eth0");
        // single con with no device
        let con2 = NMCon::from_con_list_line(
            "Wired connection 2:cbb9ec82-6d46-3b2b-921c-686714565946:802-3-ethernet::\n",
        )?;
        assert_eq!(con2.name, "Wired connection 2");
        assert!(con2.device.is_none());
        // created with `nmcli c add type dummy con-name "escape test:\\!'\"\$@, ok"  ifname d3`
        let escape = NMCon::from_con_list_line(
            r#"escape test\:\\!'"$@, ok:2f6fcae6-fcf0-4db7-ac7f-59823afd5bf4:dummy:d3:activated"#,
        )?;
        assert_eq!(escape.name, r#"escape test:\!'"$@, ok"#);
        // single con parse error
        assert!(NMCon::from_con_list_line("Wired connection 2:cbb9ec8").is_err());
        // map test
        let con_vec = nmconvec_from_con_list(
            r#"
Wired connection 1:18d241f1-946c-3325-974f-65cda3e6eea5:802-3-ethernet:eth0:active
Wired connection 2:cbb9ec82-6d46-3b2b-921c-686714565946:802-3-ethernet::
"#,
        )?;
        assert_eq!(con_vec[0], con1);
        assert_eq!(con_vec[1], con2);
        // map error
        assert!(nmconvec_from_con_list(
            r#"
Wired connection 1:18d241f1-946c-3325-974f-65cda3e6eea5:802-3-ethernet:eth0:active
Wired connection 2:cbb9ec8
"#
        )
        .is_err());
        Ok(())
    }
    #[test]
    fn test_parse_show() -> Result<()> {
        let con = NMCon::from_con_show(
            r#"
connection.id:Wired connection 1
connection.uuid:18d241f1-946c-3325-974f-65cda3e6eea5
connection.stable-id:
connection.type:802-3-ethernet
connection.interface-name:eth0
connection.autoconnect:yes
connection.autoconnect-priority:-999
connection.autoconnect-retries:-1
connection.multi-connect:0
connection.auth-retries:-1
connection.timestamp:1686553873
connection.read-only:no
connection.permissions:
connection.zone:
connection.master:
connection.slave-type:
connection.autoconnect-slaves:-1
connection.secondaries:
connection.gateway-ping-timeout:0
connection.metered:unknown
connection.lldp:default
connection.mdns:-1
connection.llmnr:-1
connection.dns-over-tls:-1
connection.mptcp-flags:0x0
connection.wait-device-timeout:-1
connection.wait-activation-delay:-1
802-3-ethernet.port:
802-3-ethernet.speed:0
802-3-ethernet.duplex:
802-3-ethernet.auto-negotiate:no
802-3-ethernet.mac-address:
802-3-ethernet.cloned-mac-address:
802-3-ethernet.generate-mac-address-mask:
802-3-ethernet.mac-address-blacklist:
802-3-ethernet.mtu:auto
802-3-ethernet.s390-subchannels:
802-3-ethernet.s390-nettype:
802-3-ethernet.s390-options:
802-3-ethernet.wake-on-lan:default
802-3-ethernet.wake-on-lan-password:
802-3-ethernet.accept-all-mac-addresses:-1
ipv4.method:auto
ipv4.dns:
ipv4.dns-search:
ipv4.dns-options:
ipv4.dns-priority:0
ipv4.addresses:
ipv4.gateway:
ipv4.routes:
ipv4.route-metric:-1
ipv4.route-table:0
ipv4.routing-rules:
ipv4.ignore-auto-routes:no
ipv4.ignore-auto-dns:no
ipv4.dhcp-client-id:
ipv4.dhcp-iaid:
ipv4.dhcp-timeout:0
ipv4.dhcp-send-hostname:yes
ipv4.dhcp-hostname:
ipv4.dhcp-fqdn:
ipv4.dhcp-hostname-flags:0x0
ipv4.never-default:no
ipv4.may-fail:yes
ipv4.required-timeout:-1
ipv4.dad-timeout:-1
ipv4.dhcp-vendor-class-identifier:
ipv4.link-local:0
ipv4.dhcp-reject-servers:
ipv6.method:auto
ipv6.dns:
ipv6.dns-search:
ipv6.dns-options:
ipv6.dns-priority:0
ipv6.addresses:
ipv6.gateway:
ipv6.routes:
ipv6.route-metric:-1
ipv6.route-table:0
ipv6.routing-rules:
ipv6.ignore-auto-routes:no
ipv6.ignore-auto-dns:no
ipv6.never-default:no
ipv6.may-fail:yes
ipv6.required-timeout:-1
ipv6.ip6-privacy:-1
ipv6.addr-gen-mode:eui64
ipv6.ra-timeout:0
ipv6.mtu:auto
ipv6.dhcp-duid:
ipv6.dhcp-iaid:
ipv6.dhcp-timeout:0
ipv6.dhcp-send-hostname:yes
ipv6.dhcp-hostname:
ipv6.dhcp-hostname-flags:0x0
ipv6.token:
proxy.method:none
proxy.browser-only:no
proxy.pac-url:
proxy.pac-script:
GENERAL.NAME:Wired connection 1
GENERAL.UUID:18d241f1-946c-3325-974f-65cda3e6eea5
GENERAL.DEVICES:eth0
GENERAL.IP-IFACE:eth0
GENERAL.STATE:activated
GENERAL.DEFAULT:yes
GENERAL.DEFAULT6:no
GENERAL.SPEC-OBJECT:
GENERAL.VPN:no
GENERAL.DBUS-PATH:/org/freedesktop/NetworkManager/ActiveConnection/18
GENERAL.CON-PATH:/org/freedesktop/NetworkManager/Settings/1
GENERAL.ZONE:
GENERAL.MASTER-PATH:
IP4.ADDRESS[1]:10.53.123.2/32
IP4.ADDRESS[2]:172.16.1.54/16
IP4.GATEWAY:172.16.0.1
IP4.ROUTE[1]:dst = 0.0.0.0/0, nh = 172.16.0.1, mt = 100
IP4.ROUTE[2]:dst = 172.16.0.0/16, nh = 0.0.0.0, mt = 100
IP4.DNS[1]:192.168.10.1
IP4.DNS[2]:192.168.10.2
IP4.DOMAIN[1]:atmark.tech
DHCP4.OPTION[1]:dhcp_client_identifier = 01:00:11:0c:00:0b:13
DHCP4.OPTION[2]:dhcp_lease_time = 28800
DHCP4.OPTION[3]:dhcp_server_identifier = 172.16.0.1
DHCP4.OPTION[4]:domain_name = atmark.tech
DHCP4.OPTION[5]:domain_name_servers = 192.168.10.1 192.168.10.2
DHCP4.OPTION[6]:expiry = 1686813073
DHCP4.OPTION[7]:ip_address = 172.16.1.54
DHCP4.OPTION[8]:ntp_servers = 192.168.10.1 192.168.10.2
DHCP4.OPTION[9]:requested_broadcast_address = 1
DHCP4.OPTION[10]:requested_domain_name = 1
DHCP4.OPTION[11]:requested_domain_name_servers = 1
DHCP4.OPTION[12]:requested_domain_search = 1
DHCP4.OPTION[13]:requested_host_name = 1
DHCP4.OPTION[14]:requested_interface_mtu = 1
DHCP4.OPTION[15]:requested_ms_classless_static_routes = 1
DHCP4.OPTION[16]:requested_nis_domain = 1
DHCP4.OPTION[17]:requested_nis_servers = 1
DHCP4.OPTION[18]:requested_ntp_servers = 1
DHCP4.OPTION[19]:requested_rfc3442_classless_static_routes = 1
DHCP4.OPTION[20]:requested_root_path = 1
DHCP4.OPTION[21]:requested_routers = 1
DHCP4.OPTION[22]:requested_static_routes = 1
DHCP4.OPTION[23]:requested_subnet_mask = 1
DHCP4.OPTION[24]:requested_time_offset = 1
DHCP4.OPTION[25]:requested_wpad = 1
DHCP4.OPTION[26]:routers = 172.16.0.1
DHCP4.OPTION[27]:subnet_mask = 255.255.0.0
IP6.ADDRESS[1]:fe80::211:cff:fe00:b13/64
IP6.GATEWAY:
IP6.ROUTE[1]:dst = fe80::/64, nh = ::, mt = 1024
IP6.DNS[1]:fe80::ea9f:80ff:fed7:40d9
"#,
        )?;
        assert_eq!(con.name, "Wired connection 1");
        assert_eq!(con.device.as_ref().unwrap(), "eth0");
        assert_eq!(con.prop("IP4.GATEWAY").unwrap(), "172.16.0.1");
        assert!(con.prop("IP6.GATEWAY").unwrap().is_empty());
        assert!(con.prop("non.existent.key").is_none());
        // escape test
        let con2 = NMCon::from_con_show(
            r#"
connection.id:escape test:\!'"$@, ok
connection.uuid:f0a7f8d6-dbff-4419-a5a4-c1f44e45778c
connection.stable-id:
connection.type:dummy
"#,
        )?;
        assert_eq!(con2.name, r#"escape test:\!'"$@, ok"#);
        // parse error: missing mandatory fields (No connection.id in device)
        assert!(NMCon::from_con_show("Wired connection 2:cbb9ec8").is_err());
        // parse error: Line did not contain ':'
        assert!(NMCon::from_con_show(
            r#"
Wired conneccbb9ec8
connection.id:test
connection.uuid:f0a7f8d6-dbff-4419-a5a4-c1f44e45778c
connection.stable-id:
connection.type:dummy
"#
        )
        .is_err());
        Ok(())
    }
    #[test]
    fn test_parse_show_dev() -> Result<()> {
        let con = NMCon::from_con_show(
            r#"
GENERAL.DEVICE:br_ap
GENERAL.TYPE:bridge
GENERAL.HWADDR:00:00:00:00:00:00
GENERAL.MTU:1500
GENERAL.STATE:100 (connected)
GENERAL.CONNECTION:abos_web_br_ap
GENERAL.CON-PATH:/org/freedesktop/NetworkManager/ActiveConnection/7
IP4.ADDRESS[1]:192.168.1.1/24
IP4.GATEWAY:
IP4.ROUTE[1]:dst = 192.168.1.0/24, nh = 0.0.0.0, mt = 425
IP6.GATEWAY:
"#,
        )?;
        println!("{con:?}");
        assert_eq!(con.name, "abos_web_br_ap");
        assert_eq!(con.prop("IP4.ADDRESS[1]").unwrap(), "192.168.1.1/24");
        assert!(con.prop("non.existent.key").is_none());
        // parse error
        Ok(())
    }
}
